<?php

declare(strict_types=1);

namespace Drupal\Tests\cg\Unit\Event;

use Drupal\cg\ContentGuideEvents;
use Drupal\cg\EventDispatcher\AlterFieldSettingsEvent;
use Drupal\cg\EventDispatcher\AlterFieldSettingsEventInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Tests altering display options.
 */
#[CoversClass(AlterFieldSettingsEvent::class)]
#[CoversMethod(AlterFieldSettingsEvent::class, 'getFieldSettings')]
#[Group('cg')]
class AlterFieldSettingsEventTest extends UnitTestCase {

  /**
   * Ensures a subscriber can alter the field settings.
   */
  public function testSubscriberAltersFieldSettings(): void {
    // Start with an initial settings.
    $settings = [
      'attach_selector' => '.form-item__label',
      'field_selector' => '.field--name-test',
      'field_name' => 'test',
      'display_type' => 'description',
      'document_path' => 'test.md',
    ];

    // Create event with initial options.
    $event = new AlterFieldSettingsEvent(
      elements: [],
      fieldSettings: $settings,
      items: NULL,
      settingsAdditional: [],
      widget: NULL,
    );

    // Register a test subscriber that changes the display type.
    $dispatcher = new EventDispatcher();
    $dispatcher->addSubscriber(new class() implements EventSubscriberInterface {

      /**
       * {@inheritdoc}
       */
      public static function getSubscribedEvents(): array {
        return [
          ContentGuideEvents::ALTER_FIELD_SETTINGS => 'onAlterFieldSettings',
        ];
      }

      /**
       * Change display type to "tooltip".
       *
       * @param \Drupal\cg\EventDispatcher\AlterFieldSettingsEventInterface $event
       *   The event carrying the display options array.
       */
      public function onAlterFieldSettings(AlterFieldSettingsEventInterface $event): void {
        $field_settings =& $event->getFieldSettings();
        $field_settings['display_type'] = 'tooltip';
      }

    });

    $dispatcher->dispatch($event, ContentGuideEvents::ALTER_FIELD_SETTINGS);

    // Verify the event contains the changed display type.
    $field_settings = $event->getFieldSettings();
    self::assertSame('tooltip', $field_settings['display_type']);
  }

}
