<?php

namespace Drupal\Tests\charts_plotly\Kernel;

use Drupal\Tests\charts\Kernel\ChartsKernelTestBase;

/**
 * Test the Plotly plugin's render-time behavior.
 *
 * @group charts
 */
class PlotlyRenderTest extends ChartsKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'charts',
    'charts_plotly',
  ];

  /**
   * A base chart element for reuse in tests.
   *
   * @var array
   */
  protected $baseElement;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->baseElement = [
      '#type' => 'chart',
      '#chart_type' => 'line',
      '#chart_library' => 'plotly',
      'series' => [
        '#type' => 'chart_data',
        '#title' => 'Test Series',
        '#data' => [10, 20, 30],
      ],
      'xaxis' => [
        '#type' => 'chart_xaxis',
        '#labels' => ['A', 'B', 'C'],
      ],
    ];
  }

  /**
   * Test raw options set on the main chart element.
   */
  public function testRawOptions() {
    $element = $this->baseElement;
    $element['#raw_options'] = [
      'layout' => [
        'showlegend' => FALSE,
        'paper_bgcolor' => '#eeeeee',
      ],
    ];

    // Test that 'showlegend' is correctly overridden.
    $path = ['layout', 'showlegend'];
    $this->assertJsonPropertyHasValue($element, $path, FALSE);

    // Test that 'paper_bgcolor' is merged.
    $path = ['layout', 'paper_bgcolor'];
    $this->assertJsonPropertyHasValue($element, $path, '#eeeeee');

    // Test raw options on a data series.
    $element['series']['#raw_options'] = [
      'line' => ['shape' => 'spline'],
    ];
    $path = ['data', 0, 'line', 'shape'];
    $this->assertJsonPropertyHasValue($element, $path, 'spline');
  }

  /**
   * Test the chart type conversion logic.
   */
  public function testChartTypeConversion() {
    // Test 'line' conversion.
    $element = $this->baseElement;
    $element['#chart_type'] = 'line';
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'scatter');
    $path = ['data', 0, 'mode'];
    $this->assertJsonPropertyHasValue($element, $path, 'lines');

    // Test 'column' conversion.
    $element['#chart_type'] = 'column';
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'bar');

    // Test 'donut' conversion.
    $element['#chart_type'] = 'donut';
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'pie');
    $path = ['data', 0, 'hole'];
    $this->assertJsonPropertyHasValue($element, $path, 0.4);

    // Test 'gauge' conversion.
    $element['#chart_type'] = 'gauge';
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'indicator');
    $path = ['data', 0, 'mode'];
    $this->assertJsonPropertyHasValue($element, $path, 'gauge+number');
  }

  /**
   * Test polar chart (radar) rendering.
   */
  public function testPolarChart() {
    $element = $this->baseElement;
    $element['#chart_type'] = 'line';
    $element['#polar'] = TRUE;
    $element['#polar_max'] = 100;
    $element['#polar_min'] = 0;

    // Test that the data type is 'scatterpolar'.
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'scatterpolar');

    // Test that polar layout is created.
    $path = ['layout', 'polar', 'radialaxis', 'range', 0];
    $this->assertJsonPropertyHasValue($element, $path, 0);
    $path = ['layout', 'polar', 'radialaxis', 'range', 1];
    $this->assertJsonPropertyHasValue($element, $path, 100);
  }

  /**
   * Test heatmap data aggregation.
   */
  public function testHeatmapData() {
    $element = [
      '#type' => 'chart',
      '#chart_type' => 'heatmap',
      '#chart_library' => 'plotly',
      'xaxis' => [
        '#type' => 'chart_xaxis',
        '#labels' => ['X1', 'X2'],
      ],
      'series_1' => [
        '#type' => 'chart_data',
        '#title' => 'Y1',
        '#data' => [10, 20],
      ],
      'series_2' => [
        '#type' => 'chart_data',
        '#title' => 'Y2',
        '#data' => [30, 40],
      ],
    ];

    // Test that there is only one data trace and its type is 'heatmap'.
    $path = ['data', 0, 'type'];
    $this->assertJsonPropertyHasValue($element, $path, 'heatmap');

    // Test x-axis labels.
    $path = ['data', 0, 'x', 1];
    $this->assertJsonPropertyHasValue($element, $path, 'X2');

    // Test y-axis labels (proves aggregation).
    $path = ['data', 0, 'y', 1];
    $this->assertJsonPropertyHasValue($element, $path, 'Y2');

    // Test z-data (proves aggregation).
    $path = ['data', 0, 'z', 1, 0];
    $this->assertJsonPropertyHasValue($element, $path, 30);
  }

}
