<?php

namespace Drupal\charts_plotly\Plugin\chart\Library;

use Drupal\charts\Attribute\Chart;
use Drupal\charts\Plugin\chart\Library\ChartBase;
use Drupal\charts\TypeManager;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The 'Plotly.js' chart library attribute.
 */
#[Chart(
  id: "plotly",
  name: new TranslatableMarkup("Plotly"),
  types: [
    "area",
    "bar",
    "boxplot",
    "bubble",
    "candlestick",
    "column",
    "donut",
    "gauge",
    "heatmap",
    "line",
    "pie",
    "radar",
    "scatter",
    "spline",
  ]
)]
class Plotly extends ChartBase implements ContainerFactoryPluginInterface {

  /**
   * The element info manager.
   *
   * @var \Drupal\Core\Render\ElementInfoManagerInterface
   */
  protected $elementInfo;

  /**
   * The chart type manager.
   *
   * @var \Drupal\charts\TypeManager
   */
  protected $chartTypeManager;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * A map of y-axis keys to their numeric index.
   *
   * @var array
   */
  protected array $axisKeyMap = [];

  /**
   * Constructs a Plotly plugin object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Render\ElementInfoManagerInterface $element_info
   *   The element info manager.
   * @param \Drupal\charts\TypeManager $chart_type_manager
   *   The chart type manager.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ElementInfoManagerInterface $element_info, TypeManager $chart_type_manager, FormBuilderInterface $form_builder) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->elementInfo = $element_info;
    $this->chartTypeManager = $chart_type_manager;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('element_info'),
      $container->get('plugin.manager.charts_type'),
      $container->get('form_builder'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function addBaseSettingsElementOptions(array &$element, array $options, FormStateInterface $form_state, array &$complete_form = []): void {
    // Applies to all chart types.
    $element['enable_3d'] = [
      '#title' => $this->t('Enable 3D visualization'),
      '#type' => 'checkbox',
      '#default_value' => !empty($options['enable_3d']),
      '#description' => $this->t('Enable 3D visualization for supported chart types.'),
    ];

    $element['enable_dark_mode'] = [
      '#title' => $this->t('Enable "dark" theme'),
      '#type' => 'checkbox',
      '#default_value' => !empty($options['enable_dark_mode']),
      '#description' => $this->t('Enable dark mode for the charts.'),
    ];

    $element['show_modebar'] = [
      '#title' => $this->t('Show modebar'),
      '#type' => 'checkbox',
      '#default_value' => $options['show_modebar'] ?? TRUE,
      '#description' => $this->t('Show the Plotly modebar with interactive tools.'),
    ];

    if (in_array($element['#chart_type'], ['scatter', 'bubble'])) {
      $element['scatter_connect_points'] = [
        '#title' => $this->t('Connect points'),
        '#type' => 'checkbox',
        '#default_value' => !empty($options['scatter_connect_points']),
        '#description' => $this->t('If checked, scatter plot points will be connected with a line (sets mode to "lines+markers").'),
      ];
    }

    // Settings for bar and column.
    if (in_array($element['#chart_type'], ['bar', 'column'])) {
      $element['bar_mode'] = [
        '#title' => $this->t('Bar mode'),
        '#type' => 'select',
        '#options' => [
          'group' => $this->t('Group'),
          'stack' => $this->t('Stack'),
          'overlay' => $this->t('Overlay'),
          'relative' => $this->t('Relative'),
        ],
        '#default_value' => $options['bar_mode'] ?? 'group',
        '#description' => $this->t('Determines how bars at the same location appear.'),
      ];

      $element['bar_gap'] = [
        '#title' => $this->t('Bar gap'),
        '#type' => 'number',
        '#min' => 0,
        '#max' => 1,
        '#step' => 0.1,
        '#default_value' => $options['bar_gap'] ?? 0.2,
        '#description' => $this->t('Gap between bars (0-1).'),
      ];

      $element['bar_group_gap'] = [
        '#title' => $this->t('Bar group gap'),
        '#type' => 'number',
        '#min' => 0,
        '#max' => 1,
        '#step' => 0.1,
        '#default_value' => $options['bar_group_gap'] ?? 0,
        '#description' => $this->t('Gap between bar groups (0-1).'),
      ];
    }
    elseif ($element['#chart_type'] === 'line') {
      $element['line_shape'] = [
        '#title' => $this->t('Line shape'),
        '#type' => 'select',
        '#options' => [
          'linear' => $this->t('Linear'),
          'spline' => $this->t('Spline'),
          'hv' => $this->t('Horizontal then vertical'),
          'vh' => $this->t('Vertical then horizontal'),
          'hvh' => $this->t('Horizontal, vertical, horizontal'),
          'vhv' => $this->t('Vertical, horizontal, vertical'),
        ],
        '#default_value' => $options['line_shape'] ?? 'linear',
        '#description' => $this->t('Determines the line shape between points.'),
      ];

      $element['line_smoothing'] = [
        '#title' => $this->t('Line smoothing'),
        '#type' => 'number',
        '#min' => 0,
        '#max' => 1.3,
        '#step' => 0.1,
        '#default_value' => $options['line_smoothing'] ?? 1,
        '#description' => $this->t('Amount of smoothing for spline lines (0-1.3).'),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function preRender(array $element) {
    $chart_definition = [
      'data' => [],
      'layout' => [],
      'config' => [],
    ];
    $this->axisKeyMap = [];

    if (!isset($element['#id'])) {
      $element['#id'] = Html::getUniqueId('plotly-render');
    }

    $chart_definition = $this->populateOptions($element, $chart_definition);

    // Use the chart's core #polar property to determine the layout.
    if (!empty($element['#polar'])) {
      $chart_definition = $this->populatePolar($element, $chart_definition);
    }
    else {
      // Process cartesian axes for all other chart types.
      $chart_definition = $this->populateAxes($element, $chart_definition);
    }

    $chart_definition = $this->populateData($element, $chart_definition);

    $element['#attached']['library'][] = 'charts_plotly/plotly';
    $element['#attributes']['class'][] = 'charts-plotly';
    $element['#chart_definition'] = $chart_definition;

    return $element;
  }

  /**
   * Populate options.
   *
   * @param array $element
   *   The element.
   * @param array $chart_definition
   *   The chart definition.
   *
   * @return array
   *   Return the chart definition.
   */
  protected function populateOptions(array $element, array $chart_definition) {
    $chart_type = $this->chartTypeConversion($element['#chart_type']);

    // Basic chart configuration.
    $chart_definition['config']['responsive'] = TRUE;
    $chart_definition['config']['displaylogo'] = FALSE;

    // Show/hide modebar.
    if (isset($element['#library_type_options']['show_modebar'])) {
      $chart_definition['config']['displayModeBar'] = (bool) $element['#library_type_options']['show_modebar'];
    }

    // Chart dimensions.
    if (!empty($element['#width'])) {
      $chart_definition['layout']['width'] = (int) $element['#width'];
    }
    if (!empty($element['#height'])) {
      $chart_definition['layout']['height'] = (int) $element['#height'];
    }

    // Chart background.
    if (!empty($element['#background'])) {
      $chart_definition['layout']['paper_bgcolor'] = $element['#background'];
      $chart_definition['layout']['plot_bgcolor'] = $element['#background'];
    }

    // Title configuration.
    if (!empty($element['#title'])) {
      $chart_definition['layout']['title']['text'] = $element['#title'];

      // Subtitle configuration.
      if (!empty($element['#subtitle'])) {
        $chart_definition['layout']['title']['subtitle']['text'] = $element['#subtitle'];
      }

      // Title styling.
      if (!empty($element['#title_color'])) {
        $chart_definition['layout']['title']['font']['color'] = $element['#title_color'];
      }
      if (!empty($element['#title_font_size'])) {
        $chart_definition['layout']['title']['font']['size'] = (int) $element['#title_font_size'];
      }
      if (!empty($element['#font'])) {
        $chart_definition['layout']['title']['font']['family'] = $element['#font'];
      }
      if (!empty($element['#title_font_weight'])) {
        $chart_definition['layout']['title']['font']['weight'] = $element['#title_font_weight'];
      }
      if (!empty($element['#title_font_style'])) {
        $chart_definition['layout']['title']['font']['style'] = $element['#title_font_style'];
      }
    }

    // Font configuration.
    if (!empty($element['#font'])) {
      $chart_definition['layout']['font']['family'] = $element['#font'];
    }

    if (!empty($element['#font_size'])) {
      $chart_definition['layout']['font']['size'] = (int) $element['#font_size'];
    }

    // Legend configuration.
    if (isset($element['#legend'])) {
      $chart_definition['layout']['showlegend'] = (bool) $element['#legend'];

      if (!empty($element['#legend_position'])) {
        $legend_position_map = [
          'top' => ['x' => 0.5, 'y' => 1.15, 'xanchor' => 'center', 'yanchor' => 'top', 'orientation' => 'h'],
          'right' => ['x' => 1.02, 'y' => 0.5, 'xanchor' => 'left', 'yanchor' => 'middle'],
          'bottom' => ['x' => 0.5, 'y' => -0.15, 'xanchor' => 'center', 'yanchor' => 'bottom', 'orientation' => 'h'],
          'left' => ['x' => -0.02, 'y' => 0.5, 'xanchor' => 'right', 'yanchor' => 'middle'],
        ];

        $position_config = $legend_position_map[$element['#legend_position']] ?? $legend_position_map['right'];
        $chart_definition['layout']['legend'] = array_merge(
          $chart_definition['layout']['legend'] ?? [],
          $position_config
        );
      }

      if (!empty($element['#legend_font_size'])) {
        $chart_definition['layout']['legend']['font']['size'] = (int) $element['#legend_font_size'];
      }
      if (!empty($element['#legend_font_weight'])) {
        $chart_definition['layout']['legend']['font']['weight'] = $element['#legend_font_weight'];
      }
      if (!empty($element['#legend_font_style'])) {
        $chart_definition['layout']['legend']['font']['style'] = $element['#legend_font_style'];
      }
      if (!empty($element['#legend_title'])) {
        $chart_definition['layout']['legend']['title']['text'] = $element['#legend_title'];
      }
    }

    // Dark mode.
    if (!empty($element['#library_type_options']['enable_dark_mode'])) {
      $chart_definition['layout']['template'] = 'plotly_dark';
    }

    // 3D visualization.
    if (!empty($element['#library_type_options']['enable_3d']) && in_array($chart_type, ['scatter', 'bar'])) {
      $chart_definition['layout']['scene'] = [
        'xaxis' => ['title' => ''],
        'yaxis' => ['title' => ''],
        'zaxis' => ['title' => ''],
      ];
    }

    // Stacking.
    if (!empty($element['#stacking'])) {
      if (in_array($chart_type, ['bar'])) {
        $chart_definition['layout']['barmode'] = 'stack';
      }
    }

    // Bar-specific configuration.
    if (in_array($element['#chart_type'], ['bar', 'column'])) {
      if (!empty($element['#library_type_options']['bar_mode'])) {
        $chart_definition['layout']['barmode'] = $element['#library_type_options']['bar_mode'];
      }

      if (isset($element['#library_type_options']['bar_gap'])) {
        $chart_definition['layout']['bargap'] = (float) $element['#library_type_options']['bar_gap'];
      }

      if (isset($element['#library_type_options']['bar_group_gap'])) {
        $chart_definition['layout']['bargroupgap'] = (float) $element['#library_type_options']['bar_group_gap'];
      }
    }

    // Gauge specific configuration.
    if ($element['#chart_type'] === 'gauge') {
      $chart_definition['layout']['height'] = $chart_definition['layout']['height'] ?? 400;
    }

    // Merge in chart raw options.
    if (!empty($element['#raw_options'])) {
      $chart_definition = NestedArray::mergeDeepArray([
        $chart_definition,
        $element['#raw_options'],
      ]);
    }

    return $chart_definition;
  }

  /**
   * Utility to populate data.
   *
   * @param array $element
   *   The element.
   * @param array $chart_definition
   *   The chart definition.
   *
   * @return array
   *   Return the chart definition.
   */
  protected function populateData(array &$element, array $chart_definition): array {
    $categories = [];
    // Use the chart's core #polar property to determine data structure.
    $is_polar = !empty($element['#polar']);

    // Get x-axis labels if they exist (for category-based charts).
    foreach (Element::children($element) as $key) {
      if ($element[$key]['#type'] === 'chart_xaxis' && !empty($element[$key]['#labels'])) {
        $categories = $element[$key]['#labels'];
        break;
      }
    }

    // If the main chart type is 'heatmap', we must aggregate all child
    // 'chart_data' elements into a single Plotly trace.
    if ($element['#chart_type'] === 'heatmap' && !$is_polar) {
      $series = [
        'type' => 'heatmap',
        'name' => $element['#title'] ?? '',
        'x' => $categories,
        'y' => [],
        'z' => [],
      ];

      // Loop all children to find chart_data elements.
      foreach (Element::children($element) as $key) {
        if ($element[$key]['#type'] === 'chart_data') {
          // Make sure defaults are loaded.
          if (empty($element[$key]['#defaults_loaded'])) {
            $element[$key] += $this->elementInfo->getInfo($element[$key]['#type']);
          }

          // Add the Y-label (from Views grouping #title).
          $series['y'][] = $element[$key]['#title'] ?? '';

          // Add the Z-data row.
          $series['z'][] = is_array($element[$key]['#data']) ? array_values($element[$key]['#data']) : [];

          // Merge raw_options from the *first* data series we find.
          if (empty($series['#raw_options_merged']) && !empty($element[$key]['#raw_options'])) {
            $series = NestedArray::mergeDeepArray([$series, $element[$key]['#raw_options']]);
            $series['#raw_options_merged'] = TRUE;
          }
        }
      }
      unset($series['#raw_options_merged']);
      $chart_definition['data'][] = $series;
    }
    else {
      // Process data series one-by-one.
      foreach (Element::children($element) as $key) {
        if ($element[$key]['#type'] === 'chart_data') {
          $series = [];

          // Make sure defaults are loaded.
          if (empty($element[$key]['#defaults_loaded'])) {
            $element[$key] += $this->elementInfo->getInfo($element[$key]['#type']);
          }

          // Series name.
          $series['name'] = $element[$key]['#title'] ?? '';

          // Determine chart type for this series.
          $series_type_original = !empty($element[$key]['#chart_type']) ? $element[$key]['#chart_type'] : $element['#chart_type'];
          $series['type'] = $this->chartTypeConversion($series_type_original);
          if ($is_polar) {
            $series['type'] = 'scatterpolar';
          }

          // Handle orientation for horizontal bar charts.
          if ($element['#chart_type'] === 'bar') {
            $series['orientation'] = 'h';
          }

          // Colors.
          if (!empty($element[$key]['#color'])) {
            $series['marker']['color'] = $element[$key]['#color'];
          }

          // This logic links a data series to a secondary axis by
          // checking for #yaxis (integer) or #target_axis (string key).
          $axis_index = 0;
          if (isset($element[$key]['#yaxis'])) {
            $axis_index = (int) $element[$key]['#yaxis'];
          }
          elseif (isset($element[$key]['#target_axis']) && isset($this->axisKeyMap[$element[$key]['#target_axis']])) {
            $axis_index = $this->axisKeyMap[$element[$key]['#target_axis']];
          }

          if ($axis_index > 0) {
            $series['yaxis'] = 'y' . ($axis_index + 1);
          }

          // Set mode for scatter plots (line, scatter, spline, area, bubble).
          if ($series['type'] === 'scatter') {
            switch ($series_type_original) {
              case 'line':
              case 'spline':
              case 'area':
                $series['mode'] = 'lines';
                if (!empty($element['#data_markers'])) {
                  $series['mode'] = 'lines+markers';
                }
                break;

              case 'scatter':
              case 'bubble':
                $series['mode'] = 'markers';
                break;
            }

            // Handle connecting points option for scatter plots.
            $connect_points = !empty($element['#library_type_options']['scatter_connect_points']);
            if (in_array($series_type_original, ['scatter', 'bubble'])) {
              if ($connect_points) {
                $series['mode'] = 'lines+markers';
              }
              else {
                $series['mode'] = 'markers';
              }
            }
            // Handle connect nulls.
            if (!empty($element['#connect_nulls'])) {
              $series['connectgaps'] = TRUE;
            }
          }

          // Line shape for line/spline charts.
          if (in_array($series_type_original, ['line', 'spline']) && !empty($element['#library_type_options']['line_shape'])) {
            $series['line']['shape'] = $element['#library_type_options']['line_shape'];
            if (!empty($element['#library_type_options']['line_smoothing'])) {
              $series['line']['smoothing'] = (float) $element['#library_type_options']['line_smoothing'];
            }
          }
          elseif ($series_type_original === 'spline') {
            $series['line']['shape'] = 'spline';
          }

          // To disable tooltips.
          if (empty($element['#tooltips'])) {
            $series['hoverinfo'] = 'none';
          }

          // Address data labels.
          if (in_array($series_type_original, ['pie', 'donut'])) {
            if (!empty($element['#data_labels'])) {
              // Enable labels for Pie/Donut charts.
              $series['textinfo'] = 'label+percent';
              $series['insidetextorientation'] = 'radial';
            }
            else {
              // Explicitly disable labels for Pie/Donut charts.
              $series['textinfo'] = 'none';
            }
          }
          elseif (!empty($element['#data_labels'])) {
            if (in_array($series_type_original, ['bar', 'column'])) {
              // For bar/column charts, we can use `text` and `textposition`.
              $series['text'] = array_values($element[$key]['#data']);
              $series['textposition'] = 'auto';
            }
            elseif ($series['type'] === 'scatter') {
              // For scatter/line charts, add the data values as text and
              // set the mode.
              $series['text'] = array_values($element[$key]['#data']);
              $series['mode'] = ($series['mode'] ?? 'lines') . '+text';
              $series['textposition'] = 'top center';
            }
          }

          // Main data population logic.
          if ($is_polar) {
            $series['r'] = array_values($element[$key]['#data']);
            $series['theta'] = !empty($categories) ? $categories : array_keys($element[$key]['#data']);
            $series['fill'] = 'toself';
            $series['mode'] = 'lines+markers';
          }
          else {
            // Populate data based on chart type for non-polar charts.
            switch ($series_type_original) {
              case 'pie':
              case 'donut':
                $data = $element[$key]['#data'];
                if (isset($data[0]) && is_array($data[0])) {
                  // Data is in nested array format ([[label, value], ...]).
                  $series['labels'] = array_column($data, 0);
                  $series['values'] = array_column($data, 1);
                }
                else {
                  // Data is standard key/value (where keys are labels).
                  $series['labels'] = !empty($element[$key]['#labels']) ? $element[$key]['#labels'] : $categories;
                  $series['values'] = $data;
                }
                if ($series_type_original === 'donut') {
                  $series['hole'] = 0.4;
                }
                break;

              case 'gauge':
                $series['mode'] = 'gauge+number';
                $series['value'] = !empty($element[$key]['#data']) ? reset($element[$key]['#data']) : 0;
                break;

              case 'area':
                $series['fill'] = 'tozeroy';
                $series['x'] = !empty($categories) ? $categories : array_keys($element[$key]['#data']);
                $series['y'] = array_values($element[$key]['#data']);
                break;

              case 'bubble':
              case 'scatter':
                if (!empty($element['#library_type_options']['scatter_connect_points'])) {
                  $series['mode'] = 'lines+markers';
                }
                else {
                  $series['mode'] = 'markers';
                }
                $data = $element[$key]['#data'];
                if (is_array($data) && isset($data[0]) && is_array($data[0])) {
                  $series['x'] = array_column($data, 0);
                  $series['y'] = array_column($data, 1);
                  if ($series_type_original === 'bubble' && isset($data[0][2])) {
                    $series['marker']['size'] = array_column($data, 2);
                  }
                }
                else {
                  // This handles standard key/value data.
                  $series['x'] = !empty($categories) ? $categories : array_keys($data);
                  $series['y'] = array_values($data);
                }
                break;

              case 'candlestick':
                $series['x'] = !empty($categories) ? $categories : array_keys($element[$key]['#data']);
                foreach ($element[$key]['#data'] as $point) {
                  if (is_array($point) && count($point) >= 4) {
                    $series['open'][] = $point[0];
                    $series['high'][] = $point[1];
                    $series['low'][] = $point[2];
                    $series['close'][] = $point[3];
                  }
                }
                break;

              case 'boxplot':
                $series['y'] = $element[$key]['#data'];
                if (!empty($categories)) {
                  $series['x'] = $categories;
                }
                break;

              default:
                if ($element['#chart_type'] === 'bar') {
                  $series['y'] = !empty($categories) ? $categories : array_keys($element[$key]['#data']);
                  $series['x'] = array_values($element[$key]['#data']);
                }
                else {
                  $series['x'] = !empty($categories) ? $categories : array_keys($element[$key]['#data']);
                  $series['y'] = array_values($element[$key]['#data']);
                }
                break;
            }
          }

          // 3D configuration must be applied *after* data is populated.
          if (!empty($element['#library_type_options']['enable_3d'])) {
            $series['type'] = 'scatter3d';
            $series['mode'] = 'markers';
            if (isset($series['x']) && isset($series['y']) && !isset($series['z'])) {
              $series['z'] = array_fill(0, count($series['x']), 0);
            }
          }

          // Merge raw options.
          if (!empty($element[$key]['#raw_options'])) {
            $series = NestedArray::mergeDeepArray([$series, $element[$key]['#raw_options']]);
          }

          $chart_definition['data'][] = $series;
        }
      }
    }

    return $chart_definition;
  }

  /**
   * Populate axes.
   *
   * @param array $element
   *   The element.
   * @param array $chart_definition
   *   The chart definition.
   *
   * @return array
   *   Return the chart definition.
   */
  protected function populateAxes(array $element, array &$chart_definition) {
    $yaxis_count = 0;
    foreach (Element::children($element) as $key) {
      $child = &$element[$key];
      if (empty($child['#defaults_loaded'])) {
        $child += $this->elementInfo->getInfo($child['#type']);
      }

      if ($child['#type'] === 'chart_xaxis') {
        $axis = [];
        if (!empty($child['#title'])) {
          $axis['title']['text'] = $child['#title'];
        }
        if (!empty($child['#raw_options'])) {
          $axis = NestedArray::mergeDeep($axis, $child['#raw_options']);
        }
        $chart_definition['layout']['xaxis'] = $axis;
      }
      elseif ($child['#type'] === 'chart_yaxis') {
        $axis = [];
        if (!empty($child['#title'])) {
          $axis['title']['text'] = $child['#title'];
        }

        // Build the axis key map for use in populateData().
        $this->axisKeyMap[$key] = $yaxis_count;
        $axis_key_name = ($yaxis_count === 0) ? 'yaxis' : 'yaxis' . ($yaxis_count + 1);

        if ($yaxis_count > 0) {
          $axis['overlaying'] = 'y';
        }
        if (!empty($child['#opposite'])) {
          $axis['side'] = 'right';
        }

        if (!empty($child['#raw_options'])) {
          $axis = NestedArray::mergeDeep($axis, $child['#raw_options']);
        }

        $chart_definition['layout'][$axis_key_name] = $axis;
        $yaxis_count++;
      }
    }

    return $chart_definition;
  }

  /**
   * Populate polar axes for radar charts.
   *
   * @param array $element
   *   The element.
   * @param array $chart_definition
   *   The chart definition.
   *
   * @return array
   *   Return the chart definition.
   */
  protected function populatePolar(array $element, array &$chart_definition) {
    $chart_definition['layout']['polar'] = [
      'radialaxis' => [
        'visible' => TRUE,
        'autorange' => TRUE,
      ],
    ];

    if (isset($element['#polar_max'])) {
      $chart_definition['layout']['polar']['radialaxis']['range'][1] = $element['#polar_max'];
    }
    if (isset($element['#polar_min'])) {
      $chart_definition['layout']['polar']['radialaxis']['range'][0] = $element['#polar_min'];
    }

    // Merge in raw options for advanced customization.
    if (!empty($element['#raw_options']['layout']['polar'])) {
      $chart_definition['layout']['polar'] = NestedArray::mergeDeep(
        $chart_definition['layout']['polar'],
        $element['#raw_options']['layout']['polar']
      );
    }

    return $chart_definition;
  }

  /**
   * Convert module chart type conventions to Plotly conventions.
   *
   * @param string|null $chart_type
   *   The chart type.
   *
   * @return string
   *   The converted chart type.
   */
  protected function chartTypeConversion(?string $chart_type): string {
    if (empty($chart_type)) {
      return '';
    }

    $chart_type_map = [
      'area' => 'scatter',
      'boxplot' => 'box',
      'bubble' => 'scatter',
      'column' => 'bar',
      'donut' => 'pie',
      'gauge' => 'indicator',
      'line' => 'scatter',
      'radar' => 'scatterpolar',
      'scatter' => 'scatter',
      'spline' => 'scatter',
    ];

    return $chart_type_map[$chart_type] ?? $chart_type;
  }

}
