import { Plugin } from 'ckeditor5/src/core';
import { toWidget } from 'ckeditor5/src/widget';

export default class ChartTextFilterEditing extends Plugin {
  static get pluginName() {
    return 'ChartTextFilterEditing';
  }

  init() {
    console.log('ChartTextFilterEditing plugin init');

    this._defineSchema();
    this._defineConverters();
    this.constructor._injectStyles();
  }

  static _injectStyles() {
    // Inject the styles directly into the page if not already present.
    const styleId = 'charts-text-filter-styles';
    if (!document.getElementById(styleId)) {
      const style = document.createElement('style');
      style.id = styleId;
      style.textContent = `
        /* Chart placeholder styles in CKEditor */
        .ck-content .chart-placeholder {
          position: relative;
          border: 2px dashed #c4c4c4;
          background-color: #f8f8f8;
          padding: 20px;
          text-align: center;
          margin: 15px 0;
          color: #555;
          font-family: sans-serif;
          border-radius: 4px;
          min-height: 80px;
          display: flex;
          align-items: center;
          justify-content: center;
        }

        /* Hover state */
        .ck-content .chart-placeholder:hover {
          background-color: #f0f0f0;
          border-color: #999;
        }

        /* Selected state */
        .ck-content .chart-placeholder.ck-widget_selected,
        .ck-content .chart-placeholder.ck-widget_selected:hover {
          outline: 3px solid var(--ck-color-focus-border);
          border-color: transparent;
          background-color: #e8f4fd;
        }

        /* Content inside the placeholder */
        .ck-content .chart-placeholder .chart-placeholder-content {
          font-size: 14px;
          color: #666;
          pointer-events: none;
          user-select: none;
        }

        /* Widget handler */
        .ck-content .chart-placeholder .ck-widget__type-around__button {
          display: block !important;
        }

        /* Make sure the widget is properly selectable */
        .ck-content .chart-placeholder.ck-widget {
          outline-width: 0;
        }

        /* Ensure proper cursor */
        .ck-content .chart-placeholder {
          cursor: default;
        }

        /* Remove any default chart styles that might interfere */
        .ck-content chart {
          display: block;
        }
      `;
      document.head.appendChild(style);
    }
  }

  _defineSchema() {
    const { schema } = this.editor.model;

    // Register the chart element in the model.
    schema.register('chart', {
      isObject: true,
      isBlock: true,
      allowWhere: '$block',
      allowAttributes: ['data-chart-config'],
    });
  }

  _defineConverters() {
    const { editor } = this;
    const { conversion } = editor;

    // Model to view conversion (downcast).
    conversion.for('downcast').elementToElement({
      model: 'chart',
      view: (modelElement, { writer: viewWriter }) => {
        const config = modelElement.getAttribute('data-chart-config');

        // Create the main container.
        const section = viewWriter.createContainerElement('chart', {
          'data-chart-config': config,
          class: 'chart-placeholder',
        });

        // Create a placeholder div inside.
        const placeholder = viewWriter.createContainerElement('div', {
          class: 'chart-placeholder-content',
        });

        // Add placeholder text.
        const placeholderText = viewWriter.createText(
          '📊 Chart (will render on save)',
        );
        viewWriter.insert(
          viewWriter.createPositionAt(placeholder, 0),
          placeholderText,
        );
        viewWriter.insert(viewWriter.createPositionAt(section, 0), placeholder);

        return toWidget(section, viewWriter, { label: 'Chart widget' });
      },
    });

    // View to model conversion (upcast).
    conversion.for('upcast').elementToElement({
      view: {
        name: 'chart',
        attributes: {
          'data-chart-config': true,
        },
      },
      model: (viewElement, { writer: modelWriter }) => {
        const config = viewElement.getAttribute('data-chart-config');
        return modelWriter.createElement('chart', {
          'data-chart-config': config,
        });
      },
    });

    // Data downcast (for getting the HTML output).
    conversion.for('dataDowncast').elementToElement({
      model: 'chart',
      view: (modelElement, { writer: viewWriter }) => {
        const config = modelElement.getAttribute('data-chart-config');

        return viewWriter.createContainerElement('chart', {
          'data-chart-config': config,
        });
      },
    });
  }
}
