<?php

namespace Drupal\charts_text_filter\Form;

use Drupal\charts_text_filter\Ajax\InsertChartCommand;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring a chart to be embedded.
 */
class ChartConfigForm extends FormBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a ChartConfigForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'charts_text_filter_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $existing_config = NULL): array {
    // Decode existing config if we are editing a chart.
    $existing_values = $existing_config ? json_decode(urldecode($existing_config), TRUE) : [];

    // Get the default library from config.
    $config = $this->configFactory->get('charts.settings');
    $default_values = NestedArray::mergeDeep($config->get('charts_default_settings'), []);

    // Add library attachments.
    $form['#attached']['library'][] = 'charts/charts';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    // Create the charts_settings element.
    $form['chart_settings'] = [
      '#type' => 'charts_settings',
      '#title' => $this->t('Chart Configuration'),
      '#used_in' => 'basic_form',
      '#series' => TRUE,
      '#default_value' => $default_values,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'button',
      '#value' => $this->t('Insert Chart'),
      '#ajax' => [
        'callback' => '::ajaxSubmitForm',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Inserting chart...'),
        ],
      ],
    ];

    $form['actions']['cancel'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#ajax' => [
        'callback' => '::ajaxCancelForm',
        'event' => 'click',
      ],
    ];

    return $form;
  }

  /**
   * AJAX callback to handle form submission.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The Ajax response.
   */
  public function ajaxSubmitForm(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    // Get the chart configuration.
    $chart_config = $form_state->getValue('chart_settings');

    // Clean up empty values.
    $chart_config = $this->cleanConfiguration($chart_config);

    // Encode as JSON.
    $json_config = json_encode($chart_config);

    // Build the chart tag.
    $chart_html = sprintf(
      '<chart data-chart-config="%s">%s</chart>',
      Html::escape($json_config),
      $this->t('Chart will render here')
    );

    $response->addCommand(new InsertChartCommand($chart_html));
    $response->addCommand(new CloseModalDialogCommand());

    return $response;
  }

  /**
   * AJAX callback to handle form cancellation.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The Ajax response.
   */
  public function ajaxCancelForm(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $response->addCommand(new CloseModalDialogCommand());
    return $response;
  }

  /**
   * Recursively cleans a configuration array.
   *
   * Removes transient form values and empty items from a nested configuration
   * array, while preserving specific keys that are allowed to be empty.
   *
   * @param mixed $config
   *   The configuration data to clean. Can be an array or a scalar value.
   *
   * @return mixed
   *   The cleaned configuration data.
   */
  private function cleanConfiguration(mixed $config): mixed {
    if (!is_array($config)) {
      return $config;
    }

    // Keys that should always be removed (form artifacts).
    static $remove_keys = [
      'delete_column' => 1,
      'weight' => 1,
      'operations' => 1,
      '_weight' => 1,
      '_delete_column_buttons' => 1,
      '_operations' => 1,
    ];

    // Keys that should be kept even if empty (structural elements).
    static $keep_if_empty = [
      'series' => 1,
      'colors' => 1,
      'display' => 1,
      'xaxis' => 1,
      'yaxis' => 1,
    ];

    foreach ($config as $key => &$value) {
      // Always remove form-specific keys.
      if (isset($remove_keys[$key])) {
        unset($config[$key]);
        continue;
      }

      // Recursively clean arrays.
      if (is_array($value)) {
        $value = $this->cleanConfiguration($value);
      }
    }

    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Not used - handled via AJAX.
  }

}
