<?php

namespace Drupal\charts_text_filter\Plugin\Filter;

use Drupal\charts\Element\Chart;
use Drupal\Component\Utility\Html;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter to allow charts to be embedded in text.
 *
 * @Filter(
 *   id = "filter_charts_text_filter",
 *   title = @Translation("Charts text filter"),
 *   description = @Translation("Allows charts to be embedded in text."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 * )
 */
class ChartsTextFilter extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a ChartsTextFilter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RendererInterface $renderer) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);
    // Use XPath to handle potential namespacing issues and provide flexibility.
    $elements = $xpath->query('//chart');

    if ($elements->length === 0) {
      return new FilterProcessResult(Html::serialize($dom));
    }

    // Loop through the elements in reverse to avoid issues with node
    // replacement.
    for ($i = $elements->length - 1; $i >= 0; $i--) {
      /** @var \DOMElement $element */
      $element = $elements->item($i);
      $config_json = $element->getAttribute('data-chart-config');
      if (empty($config_json)) {
        continue;
      }

      $config = json_decode($config_json, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE) {
        // Skip malformed JSON.
        continue;
      }

      $id = Html::getUniqueId('chart-');
      $chart_render_array = [
        $id => Chart::buildElement($config, $id),
      ];

      $rendered_chart = $this->renderer->render($chart_render_array);
      if (!empty($rendered_chart)) {
        // Create a document fragment to hold the new nodes.
        $fragment = $dom->createDocumentFragment();

        // Create a temporary document to parse the potentially multi-element
        // HTML. This is the case of the Charts Debug setting is enabled.
        $temp_doc = new \DOMDocument();
        // Use loadHTML, which is designed for HTML snippets.
        // The @ suppresses warnings for malformed HTML fragments.
        // LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD prevents <html>
        // and <body> tags from being added.
        @$temp_doc->loadHTML($rendered_chart, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

        // Import each node from the temporary doc into the main doc
        // and append it to the fragment.
        foreach ($temp_doc->childNodes as $node) {
          $imported_node = $dom->importNode($node, TRUE);
          $fragment->appendChild($imported_node);
        }

        // Replace the original <chart> placeholder with the fragment.
        $element->parentNode->replaceChild($fragment, $element);
      }
    }

    return new FilterProcessResult(Html::serialize($dom));
  }

  /**
   * Convert a string written as an array to an array.
   *
   * For example, "['a','b','c']" becomes ['a','b','c'].
   *
   * @param string $string
   *   The string to convert.
   *
   * @return array
   *   The converted array.
   */
  private function stringToArray(string $string): array {
    $string = str_replace(['[', ']', "'"], '', $string);
    return explode(',', $string);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('renderer'));
  }

}
