<?php

namespace Drupal\charts_twig;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Class ChartsTwig Extension.
 *
 * @package Drupal\charts_twig
 */
class ChartsTwig extends AbstractExtension {

  /**
   * Constructs a new YourTwigExtension object.
   *
   * @param \Drupal\Core\Render\ElementInfoManagerInterface $elementInfo
   *   The element info service.
   */
  public function __construct(protected ElementInfoManagerInterface $elementInfo) {
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions(): array {
    return [
      new TwigFunction('chart', $this->createChart(...)),
    ];
  }

  /**
   * Returns a chart given the required parameters.
   *
   * @param string|array $definition
   *   The ID of the chart when using the old syntax, or an array of chart
   *   properties when using the new syntax.
   * @param string $chart_type
   *   The type of chart to be rendered.
   * @param string $title
   *   The title of the chart.
   * @param array $chart_data
   *   The data to be rendered in the chart.
   * @param array $xaxis
   *   The x-axis data.
   * @param array $yaxis
   *   The y-axis data.
   * @param array $options
   *   The options for the chart.
   *
   * @return array
   *   The chart element.
   */
  public function createChart(string|array $definition, string $chart_type = '', string $title = '', array $chart_data = [], array $xaxis = [], array $yaxis = [], array $options = []): array {
    if (is_array($definition)) {
      return $this->renderChart($definition);
    }

    $definition = [
      'id' => $definition,
      'chart_type' => $chart_type,
      'title' => $title,
      'series' => $chart_data,
      'raw_options' => $options,
      'axes' => [],
    ];

    // Map old xaxis/yaxis to the new 'axes' structure.
    if (!empty($xaxis)) {
      $definition['axes']['xaxis'] = $xaxis;
      $definition['axes']['xaxis']['type'] = 'chart_xaxis';
    }

    if (!empty($yaxis)) {
      $definition['axes']['yaxis'] = $yaxis;
      $definition['axes']['yaxis']['type'] = 'chart_yaxis';

      // Ensure all series target this default 'yaxis'.
      foreach ($definition['series'] as $key => $series_item) {
        if (empty($series_item['target_axis'])) {
          $definition['series'][$key]['target_axis'] = 'yaxis';
        }
      }
    }

    return $this->renderChart($definition);
  }

  /**
   * Renders a chart from a single definition array.
   *
   * @param array $definition
   *   An associative array defining the chart.
   *
   * @return array
   *   The chart render array.
   */
  private function renderChart(array $definition): array {
    $id = $definition['id'] ?? Html::getUniqueId('chart');

    // Get the allowed property keys from the services.
    $chart_props = array_keys($this->elementInfo->getInfo('chart'));
    $series_props = array_keys($this->elementInfo->getInfo('chart_data'));
    // X and Y axis elements inherit from ChartAxisBase.
    $axis_props = array_keys($this->elementInfo->getInfo('chart_xaxis'));

    $chart = [];
    $chart[$id] = [
      '#type' => 'chart',
      '#id' => $id,
      '#chart_id' => $id,
    ];
    $this->mapProperties($definition, $chart_props, $chart[$id]);

    // Add series.
    $series_data = $definition['series'] ?? [];
    foreach ($series_data as $key => $series_def) {
      $series_key = 'series_' . $key;
      $chart[$id][$series_key] = ['#type' => 'chart_data'];
      $this->mapProperties($series_def, $series_props, $chart[$id][$series_key]);
    }

    // Add all axes.
    $axes_data = $definition['axes'] ?? [];
    foreach ($axes_data as $axis_key => $axis_def) {
      $axis_type = $axis_def['type'] ?? 'chart_yaxis';
      $chart[$id][$axis_key] = ['#type' => $axis_type];
      $this->mapProperties($axis_def, $axis_props, $chart[$id][$axis_key]);
    }

    return $chart;
  }

  /**
   * Helper method to map definition properties to a render element.
   *
   * @param array $definition
   *   The source array of properties from Twig.
   * @param array $allowed_props
   *   An array of allowed property keys (e.g., '#title', '#color').
   * @param array $target_element
   *   The render array element to apply properties to (by reference).
   */
  private function mapProperties(array $definition, array $allowed_props, array &$target_element): void {
    foreach ($definition as $key => $value) {
      $prop_key = '#' . $key;

      // Check if this property is allowed for the element type.
      if (in_array($prop_key, $allowed_props)) {
        // Only filter strings.
        // Assign all other types directly.
        if (is_string($value)) {
          $target_element[$prop_key] = Xss::filter($value);
        }
        else {
          $target_element[$prop_key] = $value;
        }
      }
    }
  }

}
