/**
 * @file
 * Chrome AI Summarize block behavior.
 */

(function (Drupal, once) {
  'use strict';

  /**
   * Behavior for the Chrome AI Summarize block.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the Chrome AI Summarize behavior.
   */
  Drupal.behaviors.chromeAiSummarize = {
    attach: function (context) {
      const wrappers = once('chrome-ai-summarize', '.chrome-ai-summarize-wrapper', context);

      wrappers.forEach(async(wrapper) => {
        const button = wrapper.querySelector('.js-chrome-ai-summarize-button');
        const summaryContainer = wrapper.querySelector('.chrome-ai-summarize-summary');

        if (!button || !summaryContainer) {
          return;
        }

        // Check for Summarizer API availability.
        let summarizerAvailable = false;
        let useWindowAi = false;

        try {
          if ('ai' in window && 'summarizer' in window.ai) {
            const capabilities = await window.ai.summarizer.capabilities();
            if (capabilities.available !== 'no') {
              summarizerAvailable = true;
              useWindowAi = true;

              // If model needs to be downloaded, inform the user.
              if (capabilities.available === 'after-download') {
                button.textContent = Drupal.t('Summarize This Page (Download Required)');
              }
            }
          } else if ('Summarizer' in window) {
            // Fallback for older experimental implementations.
            const availability = await window.Summarizer.availability();
            if (availability !== 'unavailable') {
              summarizerAvailable = true;
              useWindowAi = false;
            }
          }
        } catch (e) {
          // API check failed silently.
        }

        if (!summarizerAvailable) {
          // Hide the entire wrapper if API is not available.
          wrapper.style.display = 'none';
          return;
        }

        // Show the wrapper and button since the API is available.
        wrapper.classList.add('chrome-ai-summarize-api-available');
        wrapper.style.display = 'block';

        // Detect the page language from Drupal or HTML lang attribute.
        // Only support en, es, ja - default to en for other languages.
        // Normalize language code (e.g., en-US → en).
        const detectedLanguage = document.documentElement.lang || 'en';
        const normalizedLanguage = detectedLanguage.split('-')[0].toLowerCase();
        const supportedLanguages = ['en', 'es', 'ja'];
        const pageLanguage = supportedLanguages.includes(normalizedLanguage) ? normalizedLanguage : 'en';

        // Handle button click.
        button.addEventListener('click', async() => {
          // Disable button during processing.
          button.disabled = true;
          button.setAttribute('aria-busy', 'true');
          const originalButtonText = button.textContent;
          button.textContent = Drupal.t('Summarizing...');
          summaryContainer.innerHTML = '';
          summaryContainer.classList.add('chrome-ai-summarize-loading');

          try {
            // Track if we're still downloading to avoid overwriting messages.
            let isDownloading = false;

            // Create the summarizer with options.
            const summarizerOptions = {
              type: 'key-points',
              format: 'markdown',
              length: 'medium',
              outputLanguage: pageLanguage,
              monitor(m) {
                m.addEventListener('downloadprogress', (e) => {
                  isDownloading = true;
                  const percent = Math.round(e.loaded * 100);
                  button.textContent = Drupal.t('Downloading model: @percent%', {'@percent': percent});
                });
              }
            };

            let summarizer;
            if (useWindowAi) {
              summarizer = await window.ai.summarizer.create(summarizerOptions);
            } else {
              summarizer = await window.Summarizer.create(summarizerOptions);
            }

            // After creation, if we were downloading, update the button to show it's done.
            // Otherwise, keep showing "Summarizing..."
            if (isDownloading) {
              button.textContent = Drupal.t('Summarizing...');
            }

            // Get page content using innerText for clean text.
            // Target the main content area to avoid including navigation and footer.
            const main = document.getElementById('main');
            const pageContent = main ? main.innerText : document.body.innerText;

            // Create a container for the streaming output.
            const outputDiv = document.createElement('div');
            outputDiv.className = 'chrome-ai-summarize-output';
            outputDiv.setAttribute('aria-live', 'polite');
            summaryContainer.appendChild(outputDiv);

            // Use streaming to get real-time results.
            // Each chunk contains new tokens that need to be accumulated.
            if (summarizer.summarizeStreaming) {
              const stream = summarizer.summarizeStreaming(pageContent);
              let result = '';

              for await(const chunk of stream) {
                // Accumulate chunks as they stream in.
                result += chunk;
                outputDiv.textContent = result;
              }
            } else {
              // Fallback to non-streaming if streaming is not available.
              const result = await summarizer.summarize(pageContent);
              outputDiv.textContent = result;
            }

            summaryContainer.classList.remove('chrome-ai-summarize-loading');
            button.textContent = originalButtonText;
            button.disabled = false;
            button.removeAttribute('aria-busy');

            // Cleanup the summarizer if destroy method exists.
            if (summarizer.destroy) {
              summarizer.destroy();
            }
          }
          catch (error) {
            const errorDiv = document.createElement('div');
            errorDiv.className = 'chrome-ai-summarize-error';
            errorDiv.textContent = Drupal.t('An error occurred while summarizing the page. Please try again.');
            summaryContainer.innerHTML = '';
            summaryContainer.appendChild(errorDiv);
            summaryContainer.classList.remove('chrome-ai-summarize-loading');
            button.textContent = originalButtonText;
            button.disabled = false;
            button.removeAttribute('aria-busy');
          }
        });
      });
    }
  };

})(Drupal, once);
