<?php

declare(strict_types=1);

namespace Drupal\chromeless;

use Drupal\Core\TempStore\PrivateTempStoreFactory;

use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Responsible for storing the user's preferences for this module.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @internal
 */
final class ChromelessTempstore {

  /**
   * The cache contexts that should be used on each page.
   *
   * @var string[]
   */
  public readonly array $cacheContexts;

  /**
   * Whether the module is active.
   *
   * @var bool
   */
  public readonly bool $isActive;

  /**
   * Whether the title should be displayed.
   *
   * @var bool
   */
  public readonly bool $isActiveTitle;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private readonly RequestStack $requestStack;

  /**
   * The tempstore.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  private readonly PrivateTempStoreFactory $tempStoreFactory;

  /**
   * Constructs a ChromelessTempstore object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The temp store factory.
   * @param string $active
   *   The query parameter that determines if the module is active.
   * @param string $title
   *   The query parameter that controls the title display when active.
   */
  public function __construct(RequestStack $request_stack, PrivateTempStoreFactory $temp_store_factory, string $active, string $title) {
    $this->requestStack = $request_stack;
    $this->tempStoreFactory = $temp_store_factory;

    $this->isActive = $this->getPersistentValue($active);
    $this->isActiveTitle = $this->getPersistentValue($title) && $this->isActive;

    $this->cacheContexts = [
      "url.query_args:{$active}",
      "url.query_args:{$title}",
      'session',
    ];
  }

  /**
   * Get a persistent value from the current request, or the temp store.
   *
   * @param string $name
   *   The name of the value to retrieve.
   *
   * @return bool
   *   The persistent value.
   */
  private function getPersistentValue(string $name): bool {
    $request = $this->requestStack->getCurrentRequest();
    $tempstore = $this->tempStoreFactory->get('chromeless.state');

    if ($request?->query?->has($name)) {
      $tempstore->set($name, !!(int) $request->query->get($name));
    }

    return !!$tempstore->get($name);
  }

}
