<?php

declare(strict_types=1);

namespace Drupal\Tests\chromeless\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Page display variant test.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @group chromeless
 */
class PageDisplayVariantTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'chromeless',
    'chromeless_test',
    'node',
  ];

  /**
   * The site name to be used when testing.
   *
   * @var string
   */
  protected $siteName;

  /**
   * {@inheritdoc}
   */
  protected function installParameters() {
    $parameters = parent::installParameters();
    $parameters['forms']['install_configure_form']['site_name'] = $this->siteName = $this->randomMachineName();

    return $parameters;
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType([
      'type' => 'page',
    ]);
  }

  /**
   * Tests chromeless display selection.
   */
  public function testChromelessDisplaySelection(): void {
    $assert_session = $this->assertSession();

    $this->drupalPlaceBlock('system_branding_block');
    $this->drupalPlaceBlock('page_title_block');

    $test_page = $this->createNode();
    $test_page_url = $test_page->toUrl();

    // Test regular page retrieval.
    $this->drupalGet($test_page_url);

    // Regular pages SHOULD contain the site branding and page title blocks.
    $assert_session->elementTextContains('css', 'body', $this->siteName);
    $assert_session->elementTextContains('css', 'body', $test_page->label());

    // Enable chromeless mode.
    $this->drupalGet($test_page_url->setOption('query', [
      $this->container->getParameter('chromeless.query.active') => '1',
    ]));

    // Chromeless pages SHOULD NOT contain any blocks.
    $assert_session->elementTextNotContains('css', 'body', $this->siteName);
    $assert_session->elementTextNotContains('css', 'body', $test_page->label());

    // Test chromeless mode persistence without query parameters.
    $this->drupalGet($test_page_url->setOption('query', []));

    // Chromeless pages SHOULD NOT contain any blocks.
    $assert_session->elementTextNotContains('css', 'body', $this->siteName);
    $assert_session->elementTextNotContains('css', 'body', $test_page->label());
  }

  /**
   * Tests chromeless title visibility.
   */
  public function testChromelessTitleVisibility(): void {
    $assert_session = $this->assertSession();

    $test_page = $this->createNode();
    $test_page_url = $test_page->toUrl();

    // Test chromeless page retrieval without title.
    $this->drupalGet($test_page_url->setOption('query', [
      $this->container->getParameter('chromeless.query.active') => '1',
    ]));

    // The chromeless page SHOULD NOT contain the title if not enabled.
    $assert_session->elementTextNotContains('css', 'body', $test_page->label());

    // Test chromeless page retrieval with title.
    $this->drupalGet($test_page_url->setOption('query', [
      $this->container->getParameter('chromeless.query.active') => '1',
      $this->container->getParameter('chromeless.query.title') => '1',
    ]));

    // The chromeless page SHOULD contain the title if enabled.
    $assert_session->elementTextContains('css', 'body', $test_page->label());

    // Test chromeless title visibility persistence without query parameters.
    $this->drupalGet($test_page_url->setOption('query', []));

    // The chromeless page SHOULD contain the title if enabled.
    $assert_session->elementTextContains('css', 'body', $test_page->label());
  }

  /**
   * Tests custom query parameter names.
   */
  public function testCustomQueryParameterNames(): void {
    // Ensure that random query parameter names were assigned.
    $this->assertNotEquals($this->container->getParameter('chromeless.query.active'), 'chromeless');
    $this->assertNotEquals($this->container->getParameter('chromeless.query.title'), 'title');
  }

}
