<?php

declare(strict_types=1);

namespace Drupal\chromeless\Plugin\DisplayVariant;

use Drupal\chromeless\ChromelessTempstore;

use Drupal\Core\Display\PageVariantInterface;
use Drupal\Core\Display\VariantBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a page display variant that only renders the main content.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @PageDisplayVariant(
 *   id = "chromeless_page",
 *   admin_label = @Translation("Chromeless page"),
 * )
 */
final class ChromelessPageVariant extends VariantBase implements ContainerFactoryPluginInterface, PageVariantInterface {

  /**
   * The render array representing the main content.
   *
   * @var array
   */
  protected $mainContent;

  /**
   * The chromeless tempstore.
   *
   * @var \Drupal\chromeless\ChromelessTempstore
   */
  private readonly ChromelessTempstore $chromelessTempstore;

  /**
   * The page title: a string (plain title) or a render array (formatted title).
   *
   * @var string|array
   */
  protected $title = '';

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ChromelessTempstore $chromeless_tempstore) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->chromelessTempstore = $chromeless_tempstore;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('chromeless.tempstore'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function setMainContent(array $main_content) {
    $this->mainContent = $main_content;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setTitle($title) {
    $this->title = $title;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Display messages with a fallback area.
    //
    // The fallback area ensures that Drupal's JavaScript Messages API works as
    // intended by always providing a container element for messages.
    $build['content']['messages']['#type'] = 'status_messages';
    $build['content']['messages']['#include_fallback'] = TRUE;
    $build['content']['messages']['#weight'] = -1000;

    // Place the page title.
    $build['content']['page_title']['#type'] = 'page_title';
    $build['content']['page_title']['#title'] = $this->title;
    $build['content']['page_title']['#access'] = $this->chromelessTempstore->isActiveTitle;
    $build['content']['page_title']['#weight'] = -900;

    // Place the main content render array.
    $build['content']['main_content'] = $this->mainContent;
    $build['content']['main_content']['#weight'] = -800;

    return $build;
  }

}
