<?php

declare(strict_types=1);

namespace Drupal\Tests\chromeless\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Caching test.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @group chromeless
 */
class CachingTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'chromeless',
    'chromeless_test',
    'dynamic_page_cache',
    'page_cache',
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType([
      'type' => 'page',
    ]);
  }

  /**
   * Data provider for testPersistenceCaching().
   *
   * @return string[]
   *   A list of query parameter types to test.
   */
  public static function providerPersistenceCaching(): array {
    return [
      ['chromeless.query.active'],
      ['chromeless.query.title'],
    ];
  }

  /**
   * Tests persistence caching.
   *
   * @dataProvider providerPersistenceCaching
   */
  public function testPersistenceCaching(string $type): void {
    $assert_session = $this->assertSession();

    $test_page = $this->createNode();
    $test_page_url = $test_page->toUrl();

    // Test page retrieval.
    $this->drupalGet($test_page_url->setOption('query', [
      $this->container->getParameter($type) => '1',
    ]));

    // A session SHOULD be started.
    $assert_session->responseHeaderExists('Set-Cookie');

    // All pages SHOULD vary by this module's state.
    $assert_session->responseHeaderContains('X-Drupal-Cache-Contexts', 'chromeless_state');

    // Reload the page.
    $this->drupalGet($test_page_url);

    // The response SHOULD NOT be cacheable, but there SHOULD be a cache hit.
    $assert_session->responseHeaderContains('X-Drupal-Cache', 'UNCACHEABLE');
    $assert_session->responseHeaderEquals('X-Drupal-Dynamic-Cache', 'HIT');

    // All pages SHOULD vary by this module's state.
    $assert_session->responseHeaderContains('X-Drupal-Cache-Contexts', 'chromeless_state');
  }

  /**
   * Tests regular caching.
   */
  public function testRegularCaching(): void {
    $assert_session = $this->assertSession();

    $test_page = $this->createNode();
    $test_page_url = $test_page->toUrl();

    // Test regular page retrieval.
    $this->drupalGet($test_page_url);

    // Regular pages SHOULD NOT start a session.
    $assert_session->responseHeaderDoesNotExist('Set-Cookie');

    // All pages SHOULD vary by this module's state.
    $assert_session->responseHeaderContains('X-Drupal-Cache-Contexts', 'chromeless_state');

    // Reload the page.
    $this->drupalGet($test_page_url);

    // The response SHOULD be cacheable.
    $assert_session->responseHeaderEquals('X-Drupal-Cache', 'HIT');

    // All pages SHOULD vary by this module's state.
    $assert_session->responseHeaderContains('X-Drupal-Cache-Contexts', 'chromeless_state');
  }

}
