<?php

declare(strict_types=1);

namespace Drupal\chromeless\Cache;

use Drupal\chromeless\ChromelessTempstore;

use Drupal\Component\Utility\Crypt;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\Context\CacheContextInterface;

/**
 * Defines the chromeless state cache context service.
 *
 * Cache context ID: 'chromeless_state'.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @internal
 */
final class ChromelessStateCacheContext implements CacheContextInterface {

  /**
   * The chromeless tempstore.
   *
   * @var \Drupal\chromeless\ChromelessTempstore
   */
  private readonly ChromelessTempstore $chromelessTempstore;

  /**
   * Constructs a ChromelessStateCacheContext object.
   *
   * @param \Drupal\chromeless\ChromelessTempstore $chromeless_tempstore
   *   The chromeless tempstore.
   */
  public function __construct(ChromelessTempstore $chromeless_tempstore) {
    $this->chromelessTempstore = $chromeless_tempstore;
  }

  /**
   * {@inheritdoc}
   */
  public static function getLabel() {
    return \t('Chromeless state');
  }

  /**
   * {@inheritdoc}
   */
  public function getContext() {
    return Crypt::hashBase64(\json_encode([
      'active' => $this->chromelessTempstore->isActive,
      'active_title' => $this->chromelessTempstore->isActiveTitle,
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheableMetadata() {
    $cacheable_metadata = new CacheableMetadata();
    $cacheable_metadata->addCacheContexts($this->chromelessTempstore->cacheContexts);

    return $cacheable_metadata;
  }

}
