<?php

declare(strict_types=1);

namespace Drupal\chromium_tool\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Chromium Tool settings for this site.
 */
final class ChromeExecutablePathConfig extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'chromium_tool_chrome_executable_path_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['chromium_tool.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['chrome_executable_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Chromium executable path'),
      '#description' => $this->t('Absolute path to the Chromium/Chrome executable on this server.'),
      '#default_value' => (string) $this->config('chromium_tool.settings')->get('chrome_executable_path'),
      '#required' => TRUE,
      '#placeholder' => '/usr/bin/chromium',
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $path = (string) $form_state->getValue('chrome_executable_path');

    // @todo Add solution for Windows later.
    if ($path === '' || $path[0] !== '/') {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('Please provide an absolute path starting with "/".'),
      );
    }

    $resolved = is_link($path) ? realpath($path) : $path;
    if ($resolved === FALSE || !file_exists($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided path does not exist.'),
      );
    }
    elseif (!is_file($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided path is not a file.'),
      );
    }
    elseif (!is_executable($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided file is not executable.'),
      );
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('chromium_tool.settings')
      ->set('chrome_executable_path', (string) $form_state->getValue('chrome_executable_path'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
