<?php

declare(strict_types=1);

namespace Drupal\chromium_tool\Plugin\AiFunctionCall;

use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\ai\Utility\ContextDefinitionNormalizer;
use Drupal\chromium_tool\Service\ChromiumScreenshotter;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the screenshot webpage function.
 */
#[FunctionCall(
  id: 'chromium_tool:screenshot_webpage',
  function_name: 'chromium_tool_screenshot_webpage',
  name: 'Screenshot a webpage with Chromium',
  description: 'Takes a screenshot of a URL either above-the-fold (viewport) or full page.',
  group: 'browsing_tools',
  context_definitions: [
    'url' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('URL'),
      description: new TranslatableMarkup('Absolute URL to screenshot.'),
      required: TRUE,
    ),
    'mode' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('Mode'),
      description: new TranslatableMarkup('above_the_fold or full_page'),
      required: FALSE,
    ),
    'viewport_width' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Viewport width'),
      description: new TranslatableMarkup('Viewport width in pixels (default 1920).'),
      required: FALSE,
      default_value: 1920,
    ),
    'viewport_height' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Viewport height'),
      description: new TranslatableMarkup('Viewport height in pixels (default 1080).'),
      required: FALSE,
      default_value: 1080,
    ),
    'wait_ms' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Wait (ms) after load'),
      description: new TranslatableMarkup('Extra wait time after navigation for late resources.'),
      required: FALSE,
    ),
  ],
)]
final class ScreenshotWebpage extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The screenshotter service.
   *
   * @var \Drupal\chromium_tool\Service\ChromiumScreenshotter
   */
  private ChromiumScreenshotter $screenshotter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      new ContextDefinitionNormalizer(),
    );
    $instance->screenshotter = $container->get('chromium_tool.screenshotter');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $url = (string) $this->getContextValue('url');
    $mode = (string) ($this->getContextValue('mode') ?? 'above_the_fold');
    $width = (int) ($this->getContextValue('viewport_width') ?? 1920);
    $height = (int) ($this->getContextValue('viewport_height') ?? 1080);
    $waitMs = (int) ($this->getContextValue('wait_ms') ?? 500);

    if (!preg_match('@^https?://@i', $url)) {
      throw new \InvalidArgumentException('url must be an absolute http(s) URL.');
    }

    $binary = match ($mode) {
      'full_page' => $this->screenshotter->screenshotFullPage($url, $width, $height, $waitMs),
      default => $this->screenshotter->screenshotAboveTheFold($url, $width, $height, $waitMs),
    };

    $payload = [
      'mime' => 'image/png',
      'encoding' => 'base64',
      'data' => base64_encode($binary),
      'width' => $width,
      'height' => $mode === 'full_page' ? NULL : $height,
      'mode' => $mode,
      'url' => $url,
    ];
    $this->setOutput((string) json_encode($payload, JSON_UNESCAPED_SLASHES));
  }

}
