<?php

declare(strict_types=1);

namespace Drupal\chromium_tool\Plugin\AiFunctionCall;

use Drupal\ai\Attribute\FunctionCall;
use Drupal\ai\Base\FunctionCallBase;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\FunctionCallInterface;
use Drupal\chromium_tool\Service\ChromiumScreenshotter;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\image\Entity\ImageStyle;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the screenshot webpage function.
 */
#[FunctionCall(
  id: 'chromium_tool:screenshot_webpage',
  function_name: 'chromium_tool_screenshot_webpage',
  name: 'Screenshot a webpage with Chromium',
  description: 'Takes a screenshot of a URL either above-the-fold (viewport) or full page.',
  group: 'browsing_tools',
  context_definitions: [
    'url' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('URL'),
      description: new TranslatableMarkup('Absolute URL to screenshot.'),
      required: TRUE,
    ),
    'mode' => new ContextDefinition(
      data_type: 'string',
      label: new TranslatableMarkup('Mode'),
      description: new TranslatableMarkup('above_the_fold or full_page'),
      required: FALSE,
    ),
    'viewport_width' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Viewport width'),
      description: new TranslatableMarkup('Viewport width in pixels (default 1920).'),
      required: FALSE,
      default_value: 1920,
    ),
    'viewport_height' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Viewport height'),
      description: new TranslatableMarkup('Viewport height in pixels (default 1080).'),
      required: FALSE,
      default_value: 1080,
    ),
    'wait_ms' => new ContextDefinition(
      data_type: 'integer',
      label: new TranslatableMarkup('Wait (ms) after load'),
      description: new TranslatableMarkup('Extra wait time after navigation for late resources.'),
      required: FALSE,
    ),
  ],
)]
final class ScreenshotWebpage extends FunctionCallBase implements ExecutableFunctionCallInterface {

  /**
   * The screenshotter service.
   *
   * @var \Drupal\chromium_tool\Service\ChromiumScreenshotter
   */
  private ChromiumScreenshotter $screenshotter;

  /**
   * The config factory service.
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * The file system service.
   */
  private FileSystemInterface $fileSystem;

  /**
   * The logger channel factory.
   */
  private LoggerChannelFactoryInterface $loggerFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FunctionCallInterface|static {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai.context_definition_normalizer'),
      $container->get('plugin.manager.ai_data_type_converter'),
    );
    $instance->screenshotter = $container->get('chromium_tool.screenshotter');
    $instance->configFactory = $container->get('config.factory');
    $instance->fileSystem = $container->get('file_system');
    $instance->loggerFactory = $container->get('logger.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $url = (string) $this->getContextValue('url');
    $mode = (string) ($this->getContextValue('mode') ?? 'above_the_fold');
    $width = (int) ($this->getContextValue('viewport_width') ?? 1920);
    $height = (int) ($this->getContextValue('viewport_height') ?? 1080);
    $waitMs = (int) ($this->getContextValue('wait_ms') ?? 500);

    if (!preg_match('@^https?://@i', $url)) {
      throw new \InvalidArgumentException('url must be an absolute http(s) URL.');
    }

    $binary = match ($mode) {
      'full_page' => $this->screenshotter->screenshotFullPage($url, $width, $height, $waitMs),
      default => $this->screenshotter->screenshotAboveTheFold($url, $width, $height, $waitMs),
    };

    // Optionally post-process with image style if configured.
    $styleId = (string) $this->configFactory->get('chromium_tool.settings')->get('image_style');
    if ($styleId !== '') {
      try {
        /** @var \Drupal\image\Entity\ImageStyle|null $style */
        $style = ImageStyle::load($styleId);
        if ($style) {
          // Write source to a temp stream wrapper URI.
          $srcUri = 'temporary://' . uniqid('chromium_shot_', TRUE) . '.png';
          $destUri = $style->buildUri($srcUri);
          $this->fileSystem->saveData($binary, $srcUri, FileExists::Replace);
          // Generate derivative to style's destination URI.
          $ok = $style->createDerivative($srcUri, $destUri);
          if ($ok && file_exists($destUri)) {
            $processed = file_get_contents($destUri);
            if ($processed !== FALSE && $processed !== '') {
              $binary = $processed;
            }
          }
          // Cleanup temporary files if present.
          if (file_exists($srcUri)) {
            $this->fileSystem->delete($srcUri);
          }
          if (file_exists($destUri)) {
            $this->fileSystem->delete($destUri);
          }
        }
      }
      catch (\Throwable $e) {
        $this->loggerFactory->get('chromium_tool')->error('Failed to apply image style @style to screenshot: @message', [
          '@style' => $styleId,
          '@message' => $e->getMessage(),
        ]);
        // Return the original binary.
      }
    }

    $payload = [
      'mime' => 'image/png',
      'encoding' => 'base64',
      'data' => base64_encode($binary),
      'width' => $width,
      'height' => $mode === 'full_page' ? NULL : $height,
      'mode' => $mode,
      'url' => $url,
      'image_style' => $styleId !== '' ? $styleId : NULL,
    ];
    $this->setOutput((string) json_encode($payload, JSON_UNESCAPED_SLASHES));
  }

}
