<?php

declare(strict_types=1);

namespace Drupal\chromium_tool\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Chromium Tool settings for this site.
 */
final class ChromeExecutablePathConfig extends ConfigFormBase {

  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, private EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'chromium_tool_chrome_executable_path_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['chromium_tool.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['chrome_executable_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Chromium executable path'),
      '#description' => $this->t('Absolute path to the Chromium/Chrome executable on this server.'),
      '#default_value' => (string) $this->config('chromium_tool.settings')->get('chrome_executable_path'),
      '#required' => TRUE,
      '#placeholder' => '/usr/bin/chromium',
    ];

    /** @var \Drupal\image\Entity\ImageStyle[] $all */
    $all = $this->entityTypeManager->getStorage('image_style')->loadMultiple();
    $styles = [];
    foreach ($all as $id => $style) {
      $styles[$id] = $style->label();
    }

    $form['image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Image style'),
      '#description' => $this->t('If selected, screenshots will be processed with this image style.'),
      '#options' => $styles,
      '#empty_option' => $this->t('- None -'),
      '#default_value' => (string) ($this->config('chromium_tool.settings')->get('image_style') ?? ''),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $path = (string) $form_state->getValue('chrome_executable_path');

    // @todo Add solution for Windows later.
    if ($path === '' || $path[0] !== '/') {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('Please provide an absolute path starting with "/".'),
      );
    }

    $resolved = is_link($path) ? realpath($path) : $path;
    if ($resolved === FALSE || !file_exists($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided path does not exist.'),
      );
    }
    elseif (!is_file($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided path is not a file.'),
      );
    }
    elseif (!is_executable($resolved)) {
      $form_state->setErrorByName(
        'chrome_executable_path',
        $this->t('The provided file is not executable.'),
      );
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('chromium_tool.settings')
      ->set('chrome_executable_path', (string) $form_state->getValue('chrome_executable_path'))
      ->set('image_style', (string) $form_state->getValue('image_style'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
