<?php

declare(strict_types=1);

namespace Drupal\chromium_tool\Service;

use HeadlessChromium\Clip;

/**
 * Provides helpers to take webpage screenshots using chrome-php/chrome.
 */
final class ChromiumScreenshotter {

  /**
   * The browser factory wrapper that respects module configuration.
   */
  private ChromiumBrowserFactory $browserFactory;

  /**
   * Last temporary file written, used for emergency cleanup in destructor.
   */
  private ?string $lastTempFile = NULL;

  public function __construct(ChromiumBrowserFactory $browserFactory) {
    $this->browserFactory = $browserFactory;
  }

  /**
   * Ensure any stray temporary file is removed.
   */
  public function __destruct() {
    if ($this->lastTempFile && file_exists($this->lastTempFile)) {
      @unlink($this->lastTempFile);
    }
  }

  /**
   * Take an above-the-fold screenshot (viewport only).
   *
   * @param string $url
   *   Absolute URL to capture.
   * @param int $width
   *   Viewport width in pixels.
   * @param int $height
   *   Viewport height in pixels.
   * @param int $waitMs
   *   Extra wait time in milliseconds after navigation.
   *
   * @return string
   *   PNG binary.
   */
  public function screenshotAboveTheFold(string $url, int $width = 1920, int $height = 1080, int $waitMs = 500): string {
    $browser = $this->browserFactory->create()->createBrowser([
      'noSandbox' => TRUE,
      'windowSize' => [$width, $height],
    ]);
    $tmp = NULL;
    try {
      $page = $browser->createPage();
      $page->navigate($url)->waitForNavigation();
      if ($waitMs > 0) {
        usleep($waitMs * 1000);
      }
      $image = $page->screenshot([
        'format' => 'png',
        'captureBeyondViewport' => TRUE,
      ]);
      $tmp = tempnam(sys_get_temp_dir(), 'shot_') . '.png';
      $this->lastTempFile = $tmp;
      $image->saveToFile($tmp);
      $data = file_get_contents($tmp);
      if ($data === FALSE) {
        throw new \RuntimeException('Failed to read screenshot from temporary file.');
      }
      return $data;
    }
    finally {
      $browser->close();
      if ($tmp && file_exists($tmp)) {
        @unlink($tmp);
      }
      // Prevent double-cleanup by destructor.
      if ($this->lastTempFile === $tmp) {
        $this->lastTempFile = NULL;
      }
    }
  }

  /**
   * Take a full page screenshot (entire document height and width).
   *
   * @param string $url
   *   Absolute URL to capture.
   * @param int $width
   *   Initial viewport width in pixels.
   * @param int $initHeight
   *   Initial viewport height in pixels.
   * @param int $waitMs
   *   Extra wait time in milliseconds after navigation.
   *
   * @return string
   *   PNG binary.
   */
  public function screenshotFullPage(string $url, int $width = 1920, int $initHeight = 1080, int $waitMs = 500): string {
    $browser = $this->browserFactory->create()->createBrowser([
      'noSandbox' => TRUE,
      'windowSize' => [$width, $initHeight],
    ]);
    $tmp = NULL;
    try {
      $page = $browser->createPage();
      $page->navigate($url)->waitForNavigation();
      if ($waitMs > 0) {
        usleep($waitMs * 1000);
      }

      $script = <<<'JS'
(() => {
  const body = document.body;
  const html = document.documentElement;
  const height = Math.max(
    body.scrollHeight, body.offsetHeight,
    html.clientHeight, html.scrollHeight, html.offsetHeight
  );
  const width = Math.max(
    body.scrollWidth, body.offsetWidth,
    html.clientWidth, html.scrollWidth, html.offsetWidth
  );
  return { width, height };
})()
JS;
      $docMetrics = $page->evaluate($script)->getReturnValue();

      $fullWidth = (int) ($docMetrics['width'] ?? $width);
      $fullHeight = (int) ($docMetrics['height'] ?? $initHeight);

      $clip = new Clip(0, 0, $fullWidth, $fullHeight);
      $image = $page->screenshot([
        'format' => 'png',
        'captureBeyondViewport' => TRUE,
        'clip' => $clip,
      ]);
      $tmp = tempnam(sys_get_temp_dir(), 'shot_') . '.png';
      $this->lastTempFile = $tmp;
      $image->saveToFile($tmp);
      $data = file_get_contents($tmp);
      if ($data === FALSE) {
        throw new \RuntimeException('Failed to read screenshot from temporary file.');
      }
      return $data;
    }
    finally {
      $browser->close();
      if ($tmp && file_exists($tmp)) {
        @unlink($tmp);
      }
      // Prevent double-cleanup by destructor.
      if ($this->lastTempFile === $tmp) {
        $this->lastTempFile = NULL;
      }
    }
  }

}
