<?php

namespace Drupal\cilogon_globus_auth\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Admin settings for CILogon/Globus login text customization.
 */
class OSPAuthSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['cilogon_globus_auth.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'cilogon_globus_auth_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('cilogon_globus_auth.settings');

    // Determine a sensible default text format present on this site.
    $formats = \Drupal::entityTypeManager()->getStorage('filter_format')->loadMultiple();
    $default_format = 'basic_html';
    foreach (['basic_html', 'filtered_html', 'full_html'] as $candidate) {
      if (isset($formats[$candidate])) {
        $default_format = $candidate;
        break;
      }
    }

    $form['help'] = [
      '#markup' => $this->t('Customize the login button text shown on the user login page. Use @client_title placeholder to insert the OpenID Connect client label.'),
    ];

    $form['button_text_default'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default button text template'),
      '#description' => $this->t('Applied when a per‑client override is not set. Example: "Log in with @client_title"'),
      '#default_value' => $config->get('button_text_default') ?: 'Log in with @client_title',
      '#required' => TRUE,
    ];

    $help = (array) $config->get('login_help') ?: [];
    $form['login_help'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Default login help text (HTML allowed)'),
      '#description' => $this->t('Shown between the external login button(s) and the local username/password form.'),
      '#default_value' => $help['value'] ?? '',
      '#format' => $help['format'] ?? $default_format,
    ];

    // Per‑client overrides for convenience.
    $storage = \Drupal::entityTypeManager()->getStorage('openid_connect_client');
    $clients = $storage->loadMultiple();
    $overrides = (array) $config->get('button_texts') ?: [];

    if ($clients) {
      $per_help = (array) $config->get('login_help_texts') ?: [];

      $form['per_client'] = [
        '#type' => 'details',
        '#title' => $this->t('Per‑client overrides'),
        '#description' => $this->t('Customize the button text and help text per identity provider.'),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      foreach ($clients as $id => $client) {
        $form['per_client'][$id] = [
          '#type' => 'fieldset',
          '#title' => $this->t('@label', ['@label' => $client->label()]),
        ];

        $form['per_client'][$id]['button_text'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Button text override'),
          '#description' => $this->t('Leave empty to use the default template.'),
          '#default_value' => $overrides[$id] ?? '',
        ];

        $form['per_client'][$id]['help'] = [
          '#type' => 'text_format',
          '#title' => $this->t('Login help text'),
          '#description' => $this->t('HTML allowed. If empty, the global help (if any) will be used.'),
          '#default_value' => $per_help[$id]['value'] ?? '',
          '#format' => $per_help[$id]['format'] ?? $default_format,
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $values = $form_state->getValues();
    // Filter out empty overrides.
    $overrides = array_filter((array) ($values['button_texts'] ?? []), static function ($v) {
      return (string) $v !== '';
    });

    // Filter out empty per‑client help text.
    $per_help = [];
    foreach ((array) ($values['per_client'] ?? []) as $id => $item) {
      // Collect button text overrides.
      if (!empty($item['button_text'])) {
        $overrides[$id] = $item['button_text'];
      }
      // Collect per‑client help text.
      if (!empty($item['help']['value'])) {
        $per_help[$id] = [
          'value' => $item['help']['value'],
          'format' => $item['help']['format'] ?? 'filtered_html',
        ];
      }
    }

    $this->configFactory()->getEditable('cilogon_globus_auth.settings')
      ->set('button_text_default', $values['button_text_default'])
      ->set('button_texts', $overrides)
      ->set('login_help', $values['login_help'])
      ->set('login_help_texts', $per_help)
      ->save();
  }
}
