<?php

namespace Drupal\cilogon_globus_auth\Plugin\OpenIDConnectClient;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\openid_connect\Plugin\OpenIDConnectClientBase;
use Drupal\user\UserInterface;

/**
 * OSP CILogon OpenID Connect client.
 *
 * @OpenIDConnectClient(
 *   id = "ospclscigw",
 *   label = @Translation("CILogon (OSP)")
 * )
 */
class OSPCILogon extends OpenIDConnectClientBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'client_id' => '',
      // The client secret here will store a key ID when the key module is in use.
      'client_secret' => '',
      'authorization_endpoint' => 'https://cilogon.org/authorize',
      'token_endpoint' => 'https://cilogon.org/oauth2/token',
      'userinfo_endpoint' => 'https://cilogon.org/oauth2/userinfo',
      'requested_scopes' => 'email,openid,profile,org.cilogon.userinfo',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['client_id'] = [
      '#title' => $this->t('Client ID'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['client_id'],
    ];

    // Check if the key module is enabled.
    if (\Drupal::moduleHandler()->moduleExists('key')) {
      $key_collection_url = Url::fromRoute('entity.key.collection')->toString();
      $form['client_secret'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Client Secret'),
        '#empty_option' => $this->t('- Select -'),
        '#default_value' => $this->configuration['client_secret'],
        '#key_filters' => ['type' => 'authentication'],
        '#description' => $this->t('Select a key that holds the client secret. <a href=":keys">Manage keys</a>', [
          ':keys' => $key_collection_url,
        ]),
      ];
    }
    else {
      // Fallback to a plain text field if the key module is not enabled.
      $form['client_secret'] = [
        '#title' => $this->t('Client Secret'),
        '#type' => 'textfield',
        '#required' => TRUE,
        '#default_value' => $this->configuration['client_secret'],
      ];
    }

    $form['authorization_endpoint'] = [
      '#title' => $this->t('Authorization endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['authorization_endpoint'],
    ];
    $form['token_endpoint'] = [
      '#title' => $this->t('Token endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['token_endpoint'],
    ];
    $form['userinfo_endpoint'] = [
      '#title' => $this->t('UserInfo endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['userinfo_endpoint'],
    ];
    $form['requested_scopes'] = [
      '#title' => $this->t('Requested scopes'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['requested_scopes'],
    ];

    return $form;
  }

  /**
   * Exchanges the authorization code for tokens, loads user info, and maps user.
   *
   * @param string $authorization_code
   *   The authorization code from the IdP.
   * @param string $redirect_uri
   *   The redirect URI used when requesting the authorization code.
   *
   * @return \Drupal\user\UserInterface|null
   *   The authenticated Drupal user, or NULL on failure.
   */
  public function authenticate($authorization_code) {
    // Exchange the authorization code for tokens.
    $tokens = $this->retrieveTokens($authorization_code);
    if (empty($tokens) || empty($tokens['access_token'])) {
      // Authentication failed or tokens not returned.
      return NULL;
    }

    // Retrieve user info from the IdP. Adjust if your plugin has a different method.
    $user_info = $this->retrieveUserInfo($tokens['access_token']);
    if (empty($user_info) || empty($user_info['sub'])) {
      // If we can't get the user info, we can't map a Drupal user.
      return NULL;
    }

    // Build $authname (the external unique identifier).
    // Typically, "sub" is used. You might also combine it with other claims.
    $authname = $user_info['sub'];

    // Determine the idp_name. This assumes CILogon returns it in user_info['idp_name'].
    // If not, set a default or parse from other claims.
    $idp_name = !empty($user_info['idp_name']) ? $user_info['idp_name'] : 'CILogon';

    // Let the parent (or openid_connect) handle user creation or loading.
    // This maps $authname -> Drupal user.
    // The parent::authenticate() will call externalauth->save() but WITHOUT extra data.
    // So either replicate that or do a partial call and then override the authmap record.

    $drupal_user = \Drupal::currentUser();
    if (!$drupal_user instanceof UserInterface) {
      return NULL;
    }

    // Now we overwrite or add extra data in authmap. The service is "externalauth.authmap".
    // We'll store an array with 'idp_name' => $idp_name in the "data" column.
    // The provider name is "openid_connect.{plugin_id}". For your plugin: 'openid_connect.ospclscigw'
    // (or whatever your plugin ID is).
    $provider_key = 'openid_connect.' . $this->getPluginId();
    \Drupal::service('externalauth.authmap')->save(
      $drupal_user->id(),
      $provider_key,
      $authname,
      ['idp_name' => $idp_name]
    );

    return $drupal_user;
  }

  /**
   * Retrieves the client secret.
   *
   * If the key module is enabled and a key is selected, load and return its value.
   *
   * @return string
   *   The client secret.
   */
  public function getClientSecret(): string {
    if (\Drupal::moduleHandler()->moduleExists('key') && !empty($this->configuration['client_secret'])) {
      $key = \Drupal::service('key.repository')->getKey($this->configuration['client_secret']);
      if ($key) {
        return $key->getKeyValue();
      }
    }
    return $this->configuration['client_secret'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getRequestOptions(string $authorization_code, string $redirect_uri): array {
    return [
      'form_params' => [
        'code' => $authorization_code,
        'client_id' => $this->configuration['client_id'],
        // Retrieve the client secret from the key module if available.
        'client_secret' => $this->getClientSecret(),
        'redirect_uri' => $redirect_uri,
        'grant_type' => 'authorization_code',
      ],
      'headers' => [
        'Accept' => 'application/json',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoints(): array {
    return [
      'authorization' => $this->configuration['authorization_endpoint'],
      'token' => $this->configuration['token_endpoint'],
      'userinfo' => $this->configuration['userinfo_endpoint'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getClientScopes(): ?array {
    $scopes = $this->configuration['requested_scopes'];
    if (empty($scopes)) {
      return NULL;
    }
    return array_map('trim', explode(',', $scopes));
  }

}
