<?php

namespace Drupal\cilogon_globus_auth\Plugin\OpenIDConnectClient;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\openid_connect\Plugin\OpenIDConnectClientBase;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\Url;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

use Symfony\Component\HttpFoundation\RequestStack;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\openid_connect\OpenIDConnectStateTokenInterface;
use Drupal\openid_connect\OpenIDConnectAutoDiscover;

/**
 * ACCESS CI OpenID Connect client.
 *
 * @OpenIDConnectClient(
 *   id = "ospgascigw",
 *   label = @Translation("Globus Auth (OSP)")
 * )
 */
class OSPGlobusAuth extends OpenIDConnectClientBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The constructor.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The http client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Component\Datetime\TimeInterface $datetime_time
   *   The datetime.time service.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $page_cache_kill_switch
   *   Policy evaluating to static::DENY when the kill switch was triggered.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\openid_connect\OpenIDConnectStateTokenInterface $state_token
   *   The OpenID state token service.
   * @param \Drupal\openid_connect\OpenIDConnectAutoDiscover $auto_discover
   *   The OpenID well-known discovery service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\key\KeyRepositoryInterface            $key_repository
   *   The key repository service.
   */

  public function __construct(array $configuration, string $plugin_id, $plugin_definition, RequestStack $request_stack, ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory, TimeInterface $datetime_time, KillSwitch $page_cache_kill_switch, LanguageManagerInterface $language_manager, OpenIDConnectStateTokenInterface $state_token, OpenIDConnectAutoDiscover $auto_discover, ModuleHandlerInterface $module_handler, KeyRepositoryInterface $key_repository) {
    // Call the parent constructor with all required dependencies.
    parent::__construct($configuration, $plugin_id, $plugin_definition, $request_stack, $http_client, $logger_factory, $datetime_time, $page_cache_kill_switch, $language_manager, $state_token, $auto_discover);
    $this->moduleHandler = $module_handler;
    $this->keyRepository = $key_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('http_client'),
      $container->get('logger.factory'),
      $container->get('datetime.time'),
      $container->get('page_cache_kill_switch'),
      $container->get('language_manager'),
      $container->get('openid_connect.state_token'),
      $container->get('openid_connect.autodiscover'),
      $container->get('module_handler'),
      $container->get('key.repository')
    );
  }
  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'client_id' => '',
      // The client secret here will store a key ID when the key module is in use.
      'client_secret' => '',
      'authorization_endpoint' => 'https://auth.globus.org/v2/oauth2/authorize',
      'token_endpoint' => 'https://auth.globus.org/v2/oauth2/token',
      'userinfo_endpoint' => 'https://auth.globus.org/v2/oauth2/userinfo',
      'requested_scopes' => 'openid email',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['client_id'] = [
      '#title' => $this->t('Client ID'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['client_id'],
    ];

    // Check if the key module is enabled.
    if ($this->moduleHandler->moduleExists('key')) {
      $key_collection_url = Url::fromRoute('entity.key.collection')->toString();
      $form['client_secret'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Client Secret'),
        '#empty_option' => $this->t('- Select -'),
        '#default_value' => $this->configuration['client_secret'],
        '#key_filters' => ['type' => 'authentication'],
        '#description' => $this->t('Select a key that holds the client secret. <a href=":keys">Manage keys</a>', [
          ':keys' => $key_collection_url,
        ]),
      ];
    }
    else {
      // Fallback to a plain text field if the key module is not enabled.
      $form['client_secret'] = [
        '#title' => $this->t('Client Secret'),
        '#type' => 'textfield',
        '#required' => TRUE,
        '#default_value' => $this->configuration['client_secret'],
      ];
    }

    $form['authorization_endpoint'] = [
      '#title' => $this->t('Authorization endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['authorization_endpoint'],
    ];
    $form['token_endpoint'] = [
      '#title' => $this->t('Token endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['token_endpoint'],
    ];
    $form['userinfo_endpoint'] = [
      '#title' => $this->t('UserInfo endpoint'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['userinfo_endpoint'],
    ];
    $form['requested_scopes'] = [
      '#title' => $this->t('Requested scopes'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->configuration['requested_scopes'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function authorize(string $scope = 'openid email', array $additional_params = []): Response {
    $redirect_uri = $this->getRedirectUrl()->toString(TRUE);
    $url_options = $this->getUrlOptions($scope, $redirect_uri);

    $additional_params['session_required_policies'] = "9e45e494-e88b-492e-9ede-36862365b0bb";
    if (!empty($additional_params)) {
      $url_options['query'] = array_merge($url_options['query'], $additional_params);
    }

    $endpoints = $this->getEndpoints();
    // Clear _GET['destination'] because we need to override it.
    $this->requestStack->getCurrentRequest()->query->remove('destination');
    $authorization_endpoint = Url::fromUri($endpoints['authorization'], $url_options)->toString(TRUE);

    $this->loggerFactory->get('openid_connect_' . $this->pluginId)->debug('Send authorize request to @url', ['@url' => $authorization_endpoint->getGeneratedUrl()]);
    $response = new TrustedRedirectResponse($authorization_endpoint->getGeneratedUrl());
    // We can't cache the response, since this will prevent the state to be
    // added to the session. The kill switch will prevent the page getting
    // cached for anonymous users when page cache is active.
    $this->pageCacheKillSwitch->trigger();

    return $response;
  }

  /**
   * Retrieves the client secret.
   *
   * If the key module is enabled and a key is selected, load and return its value.
   *
   * @return string
   *   The client secret.
   */
  public function getClientSecret(): string {
    if ($this->moduleHandler->moduleExists('key') && !empty($this->configuration['client_secret'])) {
      $key = $this->keyRepository->getKey($this->configuration['client_secret']);
      if ($key) {
        return $key->getKeyValue();
      }
    }
    return $this->configuration['client_secret'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getRequestOptions(string $authorization_code, string $redirect_uri): array {
    return [
      'form_params' => [
        'code' => $authorization_code,
        'client_id' => $this->configuration['client_id'],
        // Retrieve the client secret from the key module if available.
        'client_secret' => $this->getClientSecret(),
        'redirect_uri' => $redirect_uri,
        'grant_type' => 'authorization_code',
      ],
      'headers' => [
        'Accept' => 'application/json',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoints(): array {
    return [
      'authorization' => $this->configuration['authorization_endpoint'],
      'token' => $this->configuration['token_endpoint'],
      'userinfo' => $this->configuration['userinfo_endpoint'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getClientScopes(): ?array {
    $scopes = $this->configuration['requested_scopes'];
    if (empty($scopes)) {
      return NULL;
    }
    return array_map('trim', explode(',', $scopes));
  }

}
