/**
 * @file
 * City Timezones behavior.
 */

(function (Drupal, once) {
  'use strict'

  /**
   * Check if the element is using Chosen library and fire update event if so.
   *
   * @param {Element} element
   */
  const updateChosenElement = function (element) {
    if (element.hasAttribute('data-once') && element.getAttribute('data-once').includes('chosen')) {
      const event = new Event('chosen:updated', { bubbles: true, cancelable: true })
      element.dispatchEvent(event)
    }
  }

  Drupal.behaviors.cityTimezones = {
    attach: function (context, settings) {
      once('init-city-timezones', '*[data-drupal-selector="edit-timezone-city"]', context).forEach(function (element) {
        // Get list of cities.
        fetch('/system/city-timezones/all')
        .then(function(res) {
          return res.json()
        })
        .then(function(data) {
          const cities = data

          // Append options.
          cities.forEach(function(city) {
            let option = document.createElement('option')
            option.value = city[0]
            option.innerHTML = city[1] + ', ' + city[10] + ', ' + city[8]
            element.appendChild(option)
          })

          // Can be Chosen element.
          updateChosenElement(element)
        })

        // When a city is chosen.
        element.addEventListener('change', function (event) {
          const id = event.currentTarget.value

          fetch('/system/city-timezones/lookup', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({
              id: id,
            })
          })
          .then(function(res) {
            return res.json()
          })
          .then(function(data) {
            const timezone = data
            const timezoneFormElement = document.querySelector('select[data-drupal-selector="edit-timezone"]')

            if (timezone && timezoneFormElement) {
              timezoneFormElement.value = timezone

              // Can be Chosen element.
              updateChosenElement(timezoneFormElement)
            }
          })
        })

        // On form submit disable city_timezone to avoid form errors.
        // Get form
        const form = element.closest('form')
        form.addEventListener('submit', function (event) {
          const timezoneCityFormElement = event.currentTarget.querySelector('select[data-drupal-selector="edit-timezone-city"]')
          if (timezoneCityFormElement) {
            timezoneCityFormElement.setAttribute('disabled', 'disabled')
          }
        })
      })
    }
  }
})(Drupal, once)
