<?php

declare(strict_types=1);

namespace Drupal\city_timezones\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Locale\CountryManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure City Timezones settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'city_timezones_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['city_timezones.settings'];
  }

  /**
   * Array options.
   *
   * @var array
   */
  protected $citySizeOptions = [
    '1000000' => '1000000',
    '100000' => '100000',
    '15000' => '15000',
    '5000' => '5000',
    '1000' => '1000',
    '500' => '500',
    'other' => 'Custom',
  ];

  /**
   * The country manager.
   *
   * @var \Drupal\Core\Locale\CountryManagerInterface
   */
  protected CountryManagerInterface $countryManager;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Locale\CountryManagerInterface $country_manager
   *   The country manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, CountryManagerInterface $country_manager) {
    parent::__construct($config_factory);
    $this->countryManager = $country_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('country_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['size'] = [
      '#type' => 'radios',
      '#title' => $this->t('Cities size'),
      '#options' => $this->citySizeOptions,
      '#default_value' => $this->config('city_timezones.settings')->get('size') ?? '15000',
      '#description' => $this->t('The minimum population of a city for it to appear in the list. The smaller the number the more cities that can be searched. The larger the number the better the performance.'),
    ];

    $form['size_custom'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom minimum population'),
      '#default_value' => $this->config('city_timezones.settings')->get('size_custom'),
      '#min' => 0,
      '#states' => [
        'visible' => [
          ':input[name="size"]' => ['value' => 'other'],
        ],
      ],
    ];

    $form['include_countries'] = [
      '#type' => 'select',
      '#title' => $this->t('Included countries'),
      '#options' => $this->countryManager->getList(),
      '#multiple' => TRUE,
      '#size' => 10,
      '#default_value' => $this->config('city_timezones.settings')->get('include_countries') ?? [],
      '#description' => $this->t('Select countries to include in the city search. If no countries are selected, all countries will be included.'),
    ];

    $form['chosen'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Chosen'),
      '#default_value' => $this->config('city_timezones.settings')->get('chosen') ?? 0,
      '#return_value' => 1,
      '#description' => $this->t('Toggles the Chosen JS library to replace the default browser select elemennt with one that can be filtered.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (in_array($form_state->getValue('size'), array_keys($this->citySizeOptions)) === FALSE) {
      $form_state->setErrorByName(
        'size',
        'This value is not correct.',
      );
    }

    $include_countries = $form_state->getValue('include_countries');
    if (!is_array($include_countries)) {
      $form_state->setErrorByName('include_countries', $this->t('The submitted country list is not valid.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('city_timezones.settings')
      ->set('size', $form_state->getValue('size'))
      ->set('size_custom', $form_state->getValue('size_custom'))
      ->set('include_countries', $form_state->getValue('include_countries'))
      ->set('chosen', $form_state->getValue('chosen'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
