<?php

declare(strict_types=1);

namespace Drupal\city_timezones\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ExtensionPathResolver;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Returns responses for City Timezones routes.
 */
final class CityTimezonesController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly ExtensionPathResolver $extensionPathResolver,
    private readonly RequestStack $requestStack,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('extension.path.resolver'),
      $container->get('request_stack'),
    );
  }

  /**
   * Builds the response.
   */
  public function all(): JsonResponse {
    $cities = $this->getCitiesFilterByCountries();

    $response = new JsonResponse($cities);

    return $response;
  }

  /**
   * Builds the response. Sends IANA timezone of the id.
   */
  public function lookup(): JsonResponse {
    // Get 'id' from request body.
    $content = $this->requestStack->getCurrentRequest()->getContent();
    $contentJson = json_decode($content, TRUE);
    $id = $contentJson['id'] ?? '';
    $city = $this->getCitiesFilterById($id);

    if (count($city) > 0) {
      $timezone = $city[17];
      $response = new JsonResponse($timezone);
    }
    else {
      $response = new JsonResponse(FALSE);
    }

    return $response;
  }

  /**
   * Get city filted by unique id.
   *
   * @return array
   *   Returns a single array with information about a single city.
   */
  private function getCitiesFilterById(string $id): array {
    $allCities = $this->getCitiesFile();

    // Convert tsv to array and create options.
    $allCitiesArray = explode(PHP_EOL, $allCities);

    foreach ($allCitiesArray as $cityString) {
      $city = preg_split("/\t/", $cityString);

      if ($city[0] == $id) {
        return $city;
      }
    }

    // Return empty array if not found.
    return [];
  }

  /**
   * Get array of cities. Optionally filter by $id.
   */
  private function getCitiesFilterByCountries(array $countries = []): array {
    $allCities = $this->getCitiesFile();

    // Convert tsv to array and create options.
    $allCitiesArray = explode(PHP_EOL, $allCities);

    $filteredCitiesArray = [];
    foreach ($allCitiesArray as $cityString) {
      $city = preg_split("/\t/", $cityString);

      if (count($countries) == 0 || isset($city[8]) && in_array($city[8], $countries)) {
        $filteredCitiesArray[] = $city;
      }
    }

    return $filteredCitiesArray;
  }

  /**
   * Get the contents of the cities file.
   */
  private function getCitiesFile(): string {
    // Open TSV file.
    $citiesPath = $this->extensionPathResolver->getPath('module', 'city_timezones');
    $resource = fopen($citiesPath . '/inc/cities5000.txt', 'r', TRUE);
    $cities5000 = fread($resource, filesize($citiesPath . '/inc/cities5000.txt'));
    fclose($resource);

    // If there is a problem return an empty string.
    if (!$cities5000) {
      return '';
    }

    return $cities5000;
  }

}
