<?php

namespace Drupal\ckeditor5_bootstrap_accordion\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;

/**
 * Defines a filter used by the Bootstrap Accordion plugin.
 *
 * @Filter(
 *   id = "filter_bootstrap_accordion",
 *   title = @Translation("Accordion enabler"),
 *   description = @Translation("Enables accordions generated by the CKEditor 5 Bootstrap Accordion plugin to work correctly by adding the attributes required by Bootstrap."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 *   weight = 10
 * )
 */
class BootstrapAccordion extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $result = new FilterProcessResult($text);

    if (stristr($text, ' data-accordion-id') === FALSE) {
      return $result;
    }

    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);

    foreach ($xpath->query("//div[@data-accordion-id][contains(concat(' ', @class, ' '), ' accordion ')]") as $accordion) {
      $accordionIdFromAttribute = $accordion->getAttribute('data-accordion-id');
      $accordionId = 'accordion-' . $accordionIdFromAttribute;
      $accordion->setAttribute('id', $accordionId);
      $accordionItemCount = 1;
      $accordionClasses = $this->getClassesFromElement($accordion);
      $accordionItemsStayOpen = in_array('accordion-items-stay-open', $accordionClasses);
      foreach ($accordion->childNodes as $accordionItem) {
        if (!($accordionItem instanceof \DOMElement)) {
          // Skips a malformed accordion item that isn't a DOMElement.
          continue;
        }
        $accordionItemClasses = $this->getClassesFromElement($accordionItem);
        if (!in_array('accordion-item', $accordionItemClasses)) {
          // Skips a malformed accordion item missing the `accordion-item`
          // class.
          continue;
        }
        $accordionCollapseId = $accordionId . '-' . ($accordionItemCount++);
        foreach ($accordionItem->childNodes as $childItem) {
          if (!($childItem instanceof \DOMElement)) {
            // Skips a malformed accordion item header or collapse that isn't a
            // DOMElement.
            continue;
          }
          $childItemClasses = $this->getClassesFromElement($childItem);
          if (in_array('accordion-header', $childItemClasses)) {
            foreach ($childItem->childNodes as $accordionButton) {
              if ($accordionButton instanceof \DOMElement && ($accordionButtonClasses = $this->getClassesFromElement($accordionButton)) && in_array('accordion-button', $accordionButtonClasses)) {
                // Adds the necessary attributes to the accordion-button.
                $accordionButton->setAttribute('role', 'button');
                $accordionButton->setAttribute('data-bs-toggle', 'collapse');
                $accordionButton->setAttribute('data-bs-target', '#' . $accordionCollapseId);
                $accordionButton->setAttribute('aria-expanded', in_array('collapsed', $accordionButtonClasses) ? 'false' : 'true');
                $accordionButton->setAttribute('aria-controls', $accordionCollapseId);
                if ($accordionButton->hasAttribute('href')) {
                  $accordionButton->setAttribute('href', '#' . $accordionCollapseId);
                }
              }
            }
          }
          elseif (in_array('accordion-collapse', $childItemClasses)) {
            // Adds the necessary attributes to the accordion-collapse.
            $childItem->setAttribute('id', $accordionCollapseId);
            if (!$accordionItemsStayOpen) {
              $childItem->setAttribute('data-bs-parent', '#' . $accordionId);
            }
          }
        }
      }
    }

    $result->setProcessedText(Html::serialize($dom));

    return $result;
  }

  /**
   * Gets the classes from a DOMElement as an array of strings.
   *
   * @param \DOMElement $element
   *   The element.
   *
   * @return string[]
   *   The element's CSS classes.
   */
  protected function getClassesFromElement(\DOMElement $element) {
    if (!$element->hasAttribute('class')) {
      return [];
    }
    return preg_split('/\s+/', $element->getAttribute('class'));
  }

}
