/*
 * Copyright (c) 2003-2024, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see https://ckeditor.com/legal/ckeditor-oss-license
 */

import CollaborationStorage
  from "../../../../../../js/ckeditor5_plugins/collaborationStorage/src/collaborationStorage";

class RealtimeAdapter {
  constructor(editor) {
    this.editor = editor;
    this.storage = new CollaborationStorage(editor);

    if (typeof drupalSettings.ckeditor5ChannelId == "undefined" ||
      typeof this.editor.sourceElement.dataset.ckeditorfieldid == "undefined" ||
      typeof drupalSettings.ckeditor5ChannelId[this.editor.sourceElement.dataset.ckeditorfieldid] == "undefined") {
      return;
    }
    this.editor.config._config.collaboration = {
      channelId: drupalSettings.ckeditor5ChannelId[this.editor.sourceElement.dataset.ckeditorfieldid],
    }
    this.setPresenceListContainer();
  }

  static get pluginName() {
    return 'RealtimeAdapter'
  }

  init() {
    const editor = this.editor;
    const hasRTC = editor.plugins.has('RealTimeCollaborativeEditing');
    const hasSourceEditing = editor.plugins.has('SourceEditing');
    if (hasRTC && hasSourceEditing) {
      console.info('The Source editing plugin is not compatible with real-time collaboration, so it has been disabled. If you need it, please contact us to discuss your use case - https://ckeditor.com/contact/');
      editor.plugins.get('SourceEditing').forceDisabled('drupal-rtc');
    }

    let editorParent = this.getFieldWrapper(this.editor.sourceElement.id);
    if (editorParent) {
      this.textFormatSelect = editorParent.querySelector(".js-filter-list");
      if (this.textFormatSelect) {
        this.textFormatSelect.addEventListener('change', this.changeEditor.bind(this));
      }
    }

  }

  getFieldWrapper(elementId) {
    const editorElement = document.getElementById(elementId);
    if (editorElement) {
      return editorElement.closest(".js-text-format-wrapper");
    }
    return null;
  }

  /**
   * Calls an endpoint that resets the collaborative session for a channel of the field that has text format changed.
   *
   * @param event
   */
  changeEditor(event) {
    const channelId = this.editor.config._config.collaboration.channelId
    const Http = new XMLHttpRequest();
    const url='/ckeditor5-premium-features-realtime-collaboration/flush-session/' + channelId;
    Http.open("DELETE", url);
    Http.send();
  }

  setPresenceListContainer() {
    const presenceListConfig = this.editor.config._config.presenceList;
    if (!presenceListConfig || typeof presenceListConfig === "undefined") {
      return;
    }
    if (!presenceListConfig.container) {
      const presenceListContainerId = this.editor.sourceElement.id + '-presence-list-container';
      const presenceListElement = document.getElementById(presenceListContainerId);
      if (!presenceListElement) {
        const formItem = this.editor.sourceElement.closest(".form-item");
        const presenceListWrapper = document.createElement("div");
        presenceListWrapper.setAttribute("class", "ck-presence-list-container");
        presenceListWrapper.setAttribute("id", presenceListContainerId);
        formItem.parentNode.insertBefore(presenceListWrapper, formItem.previousSibling);
        presenceListConfig.container = presenceListWrapper;
      } else {
        presenceListConfig.container = presenceListElement;
      }
    }

    if (!presenceListConfig.collapseAt) {
      presenceListConfig.collapseAt = drupalSettings.presenceListCollapseAt;
    }
  }

  clearPresenceListContainer() {
    const presenceListContainer = this.editor.config._config.presenceList.container;
    if (presenceListContainer) {
      presenceListContainer.innerHTML = '';
    }
  }

  /**
   * Executed after plugin is initialized.
   *
   * For the RTC it's the most suitable place to dynamically disable toolbar
   * items.
   */
  afterInit() {
    this.storage.processCollaborationCommandDisable("trackChanges");
    this.storage.processCollaborationCommandDisable("addCommentThread");
    this.checkIfInitialDataChanged();

    if (drupalSettings.ckeditor5Premium.notificationsEnabled) {
      // Hook to form submit.
      const form = this.editor.sourceElement.closest('form');
      form.addEventListener("submit", () => {
        const isCommentsEnabled = this.editor.plugins.has('CommentsRepository');
        const isTrackChangesEnabled = this.editor.plugins.has('TrackChanges');
        if (!isCommentsEnabled || !isTrackChangesEnabled) {
          return
        }

        const elementId = this.editor.sourceElement.dataset.ckeditor5PremiumElementId
        const types = {
          'trackChanges': '.track-changes',
          'comments': '.comments',
        };
        const dataAttribute = `[data-ckeditor5-premium-element-id="${elementId}"]`;

        if (isTrackChangesEnabled) {
          let trackedSuggestion = new Map()
          const trackChangesCssClass = types['trackChanges'] + '-data';
          const trackChangesPlugin = this.editor.plugins.get( 'TrackChanges' );
          const suggestions = trackChangesPlugin.getSuggestions({skipNotAttached: false});
          const trackChangesElement = document.querySelector(trackChangesCssClass + dataAttribute);
          for (let i in suggestions) {
            // Clone suggestion before adding modifications to attributes as this may break grouped suggestions.
            let clone = structuredClone(suggestions[i]);
            if (clone.head != null && (clone.next != null || clone.previous != null)) {
              clone.setAttribute('head', clone.head.id);
            }
            trackedSuggestion.set(clone.id, clone);
          }
          trackChangesElement.value = JSON.stringify(Array.from(trackedSuggestion.values()));
        }

        if (isCommentsEnabled) {
          const commentsCssClass = types['comments'] + '-data';
          const commentsRepositoryPlugin = this.editor.plugins.get( 'CommentsRepository' );
          const commentsElement = document.querySelector(commentsCssClass + dataAttribute);
          commentsElement.value = JSON.stringify(commentsRepositoryPlugin.getCommentThreads({
            skipNotAttached: true,
            skipEmpty: true,
            toJSON: true
          }));
        }
      });
    }

    this.editor.on('ready', () => {
      let textFormat = this.editor.sourceElement.dataset.editorActiveTextFormat;
      let isTrackingChangesOn = drupalSettings.ckeditor5Premium.tracking_changes.default_state;
      if (typeof isTrackingChangesOn[textFormat] !== 'undefined' && isTrackingChangesOn[textFormat]) {
        this.editor.execute('trackChanges');
      }
    });
  }

  destroy() {
    if (this.textFormatSelect || typeof this.textFormatSelect !== "undefined") {
      this.textFormatSelect.removeEventListener('change', this.changeEditor.bind(this));
    }
    this.clearPresenceListContainer();
  }

  /**
   *  Check if the editor's initial data is different from the data from CS.
   *  If so, set "data-editor-value-is-changed" attribute to TRUE.
   */
  checkIfInitialDataChanged() {
    const initialData = this.editor.config._config.initialData;
    this.editor.on('ready', () => {
      if (initialData !== this.editor.getData()) {
        this.editor.sourceElement.setAttribute('data-editor-value-is-changed', true);
      }
    } );
  }

}

export default RealtimeAdapter;
