(function (Drupal, once) {
  'use strict';

  function normalizeKey(event) {
    // Order: Cmd/Meta, Ctrl, Alt, Shift, then key.
    var parts = [];
    var isMac = /Mac|iPhone|iPad/.test(navigator.platform);

    if (event.metaKey) {
      parts.push('Cmd');
    }
    if (event.ctrlKey) {
      parts.push('Ctrl');
    }
    if (event.altKey) {
      parts.push('Alt');
    }
    if (event.shiftKey) {
      parts.push('Shift');
    }

    // De-duplicate modifiers while preserving order.
    var seen = {};
    parts = parts.filter(function (p) {
      if (seen[p]) return false;
      seen[p] = true;
      return true;
    });

    var key = '';
    // Prefer event.key for readability; fallback to code for digits.
    if (event.key && event.key.length === 1) {
      key = event.key.toUpperCase();
    } else if (event.code && /^Digit\d$/.test(event.code)) {
      key = event.code.replace('Digit', '');
    } else if (event.key && event.key.length > 1) {
      // Normalize common named keys.
      var named = event.key;
      switch (named) {
        case 'ArrowUp': key = 'ArrowUp'; break;
        case 'ArrowDown': key = 'ArrowDown'; break;
        case 'ArrowLeft': key = 'ArrowLeft'; break;
        case 'ArrowRight': key = 'ArrowRight'; break;
        case 'Escape': key = 'Esc'; break;
        case 'Enter': key = 'Enter'; break;
        case 'Tab': key = 'Tab'; break;
        case 'Backspace': key = 'Backspace'; break;
        case 'Delete': key = 'Delete'; break;
        default:
          key = named;
      }
    }

    // Ignore modifier-only presses (no non-modifier key recorded).
    if (!key || key === 'Alt' || key === 'Ctrl' || key === 'Shift' || key === 'Meta' || key === 'Cmd') {
      return '';
    }

    return parts.concat([key]).join('+');
  }

  Drupal.behaviors.ckEditor5ShortcutOverrideAdmin = {
    attach: function (context) {
      var selector = 'input[name="replacement_shortcut"]';
      once('shortcut-capture', selector, context).forEach(function (input) {
        // Improve UX: tell user to click then press.
        if (!input.placeholder) {
          input.placeholder = 'Click here, then press the desired key combination';
        }

        input.addEventListener('keydown', function (e) {
          // Allow Escape to clear.
          if (e.key === 'Escape') {
            e.preventDefault();
            e.stopPropagation();
            input.value = '';
            return;
          }

          var combo = normalizeKey(e);
          if (combo) {
            e.preventDefault();
            e.stopPropagation();
            input.value = combo;
          }
        });

        // Prevent manual text input to avoid typos.
        input.addEventListener('keypress', function (e) { e.preventDefault(); });
        input.addEventListener('paste', function (e) { e.preventDefault(); });
      });
    }
  };

})(Drupal, once);
