/**
 * @file
 * CKEditor 5 Shortcut Override plugin.
 */

(function () {
  'use strict';

  var TAG = '[ShortcutOverride]';
  var MAX_RETRIES = 10;
  var RETRY_DELAY_MS = 100;
  var REGISTERED_FLAG = '__ShortcutOverrideRegistered__';
  var SYNTHETIC_ALT0_FLAG = '__ShortcutOverrideSynthAlt0__';

  function log() {
    // Verbose logging disabled in production.
  }
  function warn() {
    if (typeof console !== 'undefined' && console.warn) {
      var args = Array.prototype.slice.call(arguments);
      args.unshift(TAG);
      try { console.warn.apply(console, args); } catch (e) {}
    }
  }
  function debug() {
    // Debug logging disabled in production.
  }

  function resolveBasePlugin() {
    try {
      if (typeof CKEditor5 === 'undefined') {
        debug('CKEditor5 is undefined at resolve time.');
        return null;
      }
      debug('CKEditor5 detected. keys:', Object.keys(CKEditor5 || {}).join(', '));

      if (CKEditor5.core && CKEditor5.core.Plugin) {
        debug('Resolved Plugin via CKEditor5.core.Plugin');
        return CKEditor5.core.Plugin;
      }
      if (CKEditor5.Plugin) {
        debug('Resolved Plugin via CKEditor5.Plugin');
        return CKEditor5.Plugin;
      }
      if (CKEditor5.dll) {
        try {
          var viaDll = CKEditor5.dll('@ckeditor/ckeditor5-core/src/plugin');
          if (viaDll) {
            debug('Resolved Plugin via CKEditor5.dll(@ckeditor/ckeditor5-core/src/plugin)');
            return viaDll;
          }
        } catch (e1) {
          debug('CKEditor5.dll path 1 failed:', e1 && e1.message ? e1.message : e1);
        }
        try {
          var viaDll2 = CKEditor5.dll('./src/plugin');
          if (viaDll2) {
            debug('Resolved Plugin via CKEditor5.dll(./src/plugin)');
            return viaDll2;
          }
        } catch (e2) {
          debug('CKEditor5.dll path 2 failed:', e2 && e2.message ? e2.message : e2);
        }
      }
    } catch (e) {
      warn('Error while resolving Base Plugin:', e && e.message ? e.message : e);
    }
    return null;
  }

  function bootstrap(attempt) {
    attempt = attempt || 1;
    debug('Bootstrap attempt', attempt);

    var BasePlugin = resolveBasePlugin();

    if (!BasePlugin) {
      if (attempt < MAX_RETRIES) {
        debug('BasePlugin not available yet. Retrying in', RETRY_DELAY_MS, 'ms');
        return setTimeout(function () { bootstrap(attempt + 1); }, RETRY_DELAY_MS);
      }
      // Fail gracefully to avoid breaking editor attachment.
      warn('CKEditor5 Plugin base not found after', attempt, 'attempts; skipping plugin registration.');
      return;
    }

    // Ensure CKEditor5 global exists; no registry needed in Drupal DLL.
    if (typeof CKEditor5 === 'undefined') {
      if (attempt < MAX_RETRIES) {
        debug('CKEditor5 global not available yet. Retrying in', RETRY_DELAY_MS, 'ms');
        return setTimeout(function () { bootstrap(attempt + 1); }, RETRY_DELAY_MS);
      }
      warn('CKEditor5 global not available after', attempt, 'attempts; plugin not registered.');
      return;
    }

    if (typeof window !== 'undefined' && window[REGISTERED_FLAG]) {
      debug('Plugin already registered earlier; skipping duplicate registration.');
      return;
    }

    /**
     * The Shortcut Override plugin.
     */
    class ShortcutOverride extends BasePlugin {
      /**
       * @inheritdoc
       */
      static get pluginName() {
        return 'ShortcutOverride';
      }

      /**
       * @inheritdoc
       */
      init() {
        var editor = this.editor;
        var config = {};
        try {
          config = editor && editor.config ? (editor.config.get('ShortcutOverride') || {}) : {};
        } catch (e) {
          warn('Error reading editor config:', e && e.message ? e.message : e);
        }

        debug('init() called. Editor available:', !!editor, 'Config:', config);

        if (!config.enabled) {
          debug('Plugin disabled via config.enabled=false; exiting init().');
          return;
        }

        // Your plugin logic here using the config values
        log('plugin initialized with config:', config);

        // Helper: resolve drupalSettings for replacement shortcut
        function getReplacementShortcutFromDrupalSettings() {
          try {
            var ds = (typeof window !== 'undefined' ? window.drupalSettings : null) || {};
            var mod = (ds && ds.ckeditor5ShortcutOverride) ? ds.ckeditor5ShortcutOverride : {};
            var rep = mod.replacementShortcut || config.replacementShortcut || '';
            if (!rep) {
              rep = 'Cmd+Alt+0';
            }
            debug('Resolved replacement shortcut from drupalSettings/config:', rep);
            return rep;
          } catch (e) {
            warn('Failed to read drupalSettings:', e && e.message ? e.message : e);
            return 'Cmd+Alt+0';
          }
        }

        // Helper: normalize human-friendly combo to CKEditor keystroke string.
        function normalizeKeystrokeString(combo) {
          if (!combo || typeof combo !== 'string') return '';
          // Trim spaces, split on '+', lower-case, map synonyms.
          var parts = combo.split('+').map(function (p) { return String(p).trim(); }).filter(Boolean);
          var key = parts.pop() || '';
          var mods = parts.map(function (m) {
            var ml = m.toLowerCase();
            if (ml === 'cmd') return 'cmd';
            if (ml === 'meta') return 'cmd';
            if (ml === 'control') return 'ctrl';
            return ml;
          });
          var k = String(key).length === 1 ? key.toLowerCase() : key;
          var normalized = mods.concat([k]).join('+');
          debug('Normalized keystroke string:', combo, '=>', normalized);
          return normalized;
        }

        // Try to open the Accessibility Help popup by executing the appropriate command.
        function executeAccessibilityHelp() {
          try {
            var names = ['openAccessibilityHelp', 'accessibilityHelp', 'a11yHelp'];
            for (var i = 0; i < names.length; i++) {
              var n = names[i];
              try {
                var cmd = editor && editor.commands && editor.commands.get ? editor.commands.get(n) : null;
                if (cmd) {
                  debug('Executing Accessibility Help command:', n);
                  editor.execute(n);
                  return true;
                }
              } catch (eInner) {
                // continue trying
              }
            }
            // Fallback: attempt UI component on editor.ui if present (last resort – may not exist)
            try {
              var ui = editor && editor.ui;
              if (ui && ui.accessibilityHelp && typeof ui.accessibilityHelp.show === 'function') {
                debug('Invoking ui.accessibilityHelp.show()');
                ui.accessibilityHelp.show();
                return true;
              }
            } catch (eUI) {}

            // Last-chance fallback: synthesize the original Alt+0 keystroke so CKEditor5's own
            // AccessibilityHelp keystroke handler opens the popup even if there is no public command.
            try {
              var targetEl = null;
              // Prefer editable element from UI if available (ClassicEditor):
              if (editor && editor.ui && typeof editor.ui.getEditableElement === 'function') {
                targetEl = editor.ui.getEditableElement();
              }
              // Otherwise try the editing view DOM root:
              if (!targetEl && editor && editor.editing && editor.editing.view && typeof editor.editing.view.getDomRoot === 'function') {
                targetEl = editor.editing.view.getDomRoot();
              }
              // Fallback: first contenteditable element inside the editor element.
              if (!targetEl && editor && editor.sourceElement && editor.sourceElement.parentNode) {
                var ce = editor.sourceElement.parentNode.querySelector('[contenteditable="true"]');
                if (ce) targetEl = ce;
              }
              if (targetEl && targetEl.dispatchEvent) {
                debug('Dispatching synthetic Alt+0 keydown to trigger Accessibility Help');
                try { if (typeof window !== 'undefined') { window[SYNTHETIC_ALT0_FLAG] = true; } } catch (eF) {}
                var evt = new KeyboardEvent('keydown', {
                  key: '0',
                  code: 'Digit0',
                  altKey: true,
                  ctrlKey: false,
                  shiftKey: false,
                  metaKey: false,
                  bubbles: true,
                  cancelable: true
                });
                targetEl.dispatchEvent(evt);
                // Clear the synthetic flag on next tick to avoid interfering with user input.
                try { if (typeof window !== 'undefined') { setTimeout(function(){ window[SYNTHETIC_ALT0_FLAG] = false; }, 0); } } catch (eC) {}
                return true;
              }
            } catch (eSynth) {
              debug('Synthetic Alt+0 dispatch failed:', eSynth && eSynth.message ? eSynth.message : eSynth);
            }

            warn('Accessibility Help command not found. Is the AccessibilityHelp plugin enabled?');
            return false;
          } catch (e) {
            warn('Error executing Accessibility Help:', e && e.message ? e.message : e);
            return false;
          }
        }

        // Implement keyboard shortcut logic: only disable the Accessibility Help alt+0 keystroke.
        try {
          var keystrokes = editor && editor.keystrokes;
          if (keystrokes && keystrokes.set) {
            // Stop CKEditor from handling Alt/Option+0 so the OS can type its character.
            keystrokes.set('alt+0', function (keyEvtData, cancel) {
              try {
                // Only intercept plain Alt+0 (no Ctrl/Cmd). If Ctrl/Meta is present, let it through.
                var hasCtrlMeta = false;
                try {
                  var de = keyEvtData && keyEvtData.domEvent ? keyEvtData.domEvent : null;
                  hasCtrlMeta = !!(de && (de.ctrlKey || de.metaKey));
                } catch (e0) {}
                if (hasCtrlMeta) {
                  debug('alt+0 with Ctrl/Meta detected — not intercepting (allowing other handlers).');
                  return;
                }
                debug('Intercepted alt+0 via keystrokes; stopping CKEditor handling without preventing default.');
                if (keyEvtData && typeof keyEvtData.stop === 'function') {
                  keyEvtData.stop();
                }
                if (keyEvtData && keyEvtData.domEvent && typeof keyEvtData.domEvent.stopImmediatePropagation === 'function') {
                  keyEvtData.domEvent.stopImmediatePropagation();
                }
                // Do not call cancel(); allow default so character is typed.
              } catch (e) {
                // ignore
              }
            }, { priority: 'high' });

            // Also add a native keydown interceptor on the editing view document as a safety net.
            try {
              if (editor.editing && editor.editing.view && editor.editing.view.document && editor.editing.view.document.on) {
                editor.editing.view.document.on('keydown', function (evt, data) {
                  try {
                    var isAlt = !!(data && (data.altKey === true));
                    // Stricter zero detection using code/key to avoid false positives.
                    var isZero = false;
                    if (typeof data.code !== 'undefined') {
                      isZero = (data.code === 'Digit0');
                    }
                    if (!isZero && typeof data.key !== 'undefined') {
                      isZero = (data.key === '0');
                    }
                    if (!isZero && typeof data.keyCode !== 'undefined') {
                      isZero = (data.keyCode === 48); // Digit0
                    }

                    if (isAlt && isZero) {
                      // If Ctrl/Cmd is held, do not treat this as the plain Alt+0 conflict — let other handlers run.
                      var hasCtrlMeta = !!(data && (data.ctrlKey || data.metaKey));
                      if (hasCtrlMeta) {
                        debug('View document alt+0 with Ctrl/Meta detected — not intercepting.');
                        return;
                      }
                      debug('View document keydown capture for alt+0: stopping internal handlers, allowing default.');
                      if (data.domEvent && typeof data.domEvent.stopImmediatePropagation === 'function') {
                        data.domEvent.stopImmediatePropagation();
                      }
                      if (evt && typeof evt.stop === 'function') {
                        evt.stop();
                      }
                    }
                  } catch (eKD) {
                    // ignore
                  }
                }, { priority: 'highest' });
                debug('Registered native keydown interceptor for alt+0 on the editing view document.');
              }
            } catch (eNat) {
              // ignore
            }
          } else {
            warn('editor.keystrokes.set not available; cannot attach key override.');
          }
        } catch (e) {
          warn('Error while wiring keystroke override:', e && e.message ? e.message : e);
        }

        // ... (rest of your plugin logic)
      }
    }

    // Expose the plugin class at the expected Drupal lookup path:
    // CKEditor5[package][Class] => CKEditor5.shortcutOverride.ShortcutOverride
    CKEditor5.shortcutOverride = CKEditor5.shortcutOverride || {};
    CKEditor5.shortcutOverride.ShortcutOverride = ShortcutOverride;

    if (typeof window !== 'undefined') {
      window[REGISTERED_FLAG] = true;
    }
    log('Plugin exposed at CKEditor5.shortcutOverride.ShortcutOverride');
  }

  // Kick off bootstrap (may retry if CKEditor5 not yet ready)
  bootstrap(1);

})();
