<?php

namespace Drupal\Tests\ckeditor5_spoiler\FunctionalJavascript;

use Drupal\Core\Session\AccountInterface;
use Drupal\editor\Entity\Editor;
use Drupal\filter\Entity\FilterFormat;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\ckeditor5\Traits\CKEditor5TestTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\RoleInterface;

/**
 * Defines tests for the ckeditor5 button and javascript functionality.
 */
class Ckeditor5SpoilerTest extends WebDriverTestBase {

  use CKEditor5TestTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'ckeditor5',
    'ckeditor5_spoiler',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The User used for the test.
   */
  protected AccountInterface $user1;

  /**
   * The User used for the test.
   */
  protected AccountInterface $user2;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page']);

    FilterFormat::create(
      [
        'format' => 'test',
        'name' => 'Ckeditor 5 with spoiler',
        'roles' => [RoleInterface::AUTHENTICATED_ID],
        'filters' => [
          'filter_html' => [
            'id' => 'filter_html',
            'status' => TRUE,
            'weight' => 2,
            'settings' => [
              'allowed_html' => '<p class> <div class>',
              'filter_html_help' => TRUE,
              'filter_html_nofollow' => FALSE,
            ],
          ],
          'filter_spoiler' => [
            'id' => 'filter_spoiler',
            'status' => TRUE,
            'weight' => 3,
          ],
        ],
      ]
    )->save();
    Editor::create(
      [
        'format' => 'test',
        'editor' => 'ckeditor5',
        'settings' => [
          'toolbar' => [
            'items' => ['Spoiler', 'sourceEditing'],
          ],
        ],
      ]
    )->save();

    $this->user1 = $this->DrupalCreateUser([
      'create page content',
      'edit own page content',
      'access content',
      'use text format test',
    ]);

    $this->user2 = $this->DrupalCreateUser([
      'access content',
      'use text format test',
    ]);
  }

  /**
   * Tests if CKEditor 5 spoilers can be interacted with in dialogs.
   */
  public function testCkeditor5Spoiler() {
    $assert_session = $this->assertSession();

    $this->drupalLogin($this->user1);

    // Add a node with text rendered via the Plain Text format.
    $this->drupalGet('node/add');

    $this->waitForEditor();
    // Ensure the editor is loaded.
    $this->click('.ck-content');

    $this->assertEditorButtonEnabled('Spoiler');
    $this->click('.ck-button');

    $assert_session->waitForElement('css', '.spoiler');

    $this->click('.ck-source-editing-button');
    $source_text_area = $assert_session->waitForElement('css', '.ck-source-editing-area textarea');
    $this->assertNotEmpty($source_text_area);

    $raw_value = $source_text_area->getValue();
    $raw_value = str_replace(["\n", "  "], '', $raw_value);

    $this->assertEquals('<div class="spoiler"><div class="spoiler-title">Spoiler title<div class="spoiler-toggle hide-icon">&nbsp;</div></div><div class="spoiler-content"><p>&nbsp;</p></div></div>', $raw_value);
  }

  /**
   * Tests if CKEditor 5 spoilers can be interacted by regular users.
   */
  public function testCkeditor5SpoilerForUsers() {
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    $node = $this->createNode([
      'type' => 'page',
      'title' => $this->randomString(),
      'body' => [
        'value' => '<div class="spoiler"><div class="spoiler-title">Spoiler title<div class="spoiler-toggle hide-icon">&nbsp;</div></div><div class="spoiler-content"><p>Spoiler content</p></div></div>',
        'format' => 'test',
      ],
    ]);

    $this->drupalLogin($this->user2);
    $this->drupalGet('node/' . $node->id());

    $spoiler_title = $page->find('css', 'div.spoiler-title');
    $this->assertNotNull($spoiler_title);
    $this->assertEquals('spoiler-title', $spoiler_title->getAttribute('data-once'));

    $style = $page->find('css', 'div.spoiler-content')->getAttribute('style');
    $this->assertNull($style);

    $spoiler_toggle = $page->find('css', 'div.spoiler-toggle');
    $spoiler_toggle->click();

    $assert_session->elementAttributeContains('css', 'div.spoiler-content', 'style', 'display: none');

    $spoiler_toggle->click();

    $assert_session->elementAttributeNotContains('css', 'div.spoiler-content', 'style', 'display: none');
  }

}
