import { Plugin } from 'ckeditor5/src/core.js';
import { aiAgentContext } from './aiagentcontext.js';
import { SUPPORTED_LANGUAGES, SHOW_ERROR_DURATION } from './const.js';
import { Widget } from 'ckeditor5/src/widget.js';
import { env } from 'ckeditor5/src/utils.js';
import { addAiAgentButton } from './util/ai-agent-button.js';
import { addAiAgentToneButton } from './util/ai-agent-tone-button.js';
import { registerInlineSlashSchema, registerAiTagSchema, registerAiAnimateStatusSchema } from './util/ai-agent-ui-schema.js';
export default class AiAgentUI extends Plugin {
    PLACEHOLDER_TEXT_ID = 'slash-placeholder';
    GPT_RESPONSE_LOADER_ID = 'gpt-response-loader';
    GPT_RESPONSE_ERROR_ID = 'gpt-error';
    showErrorDuration = SHOW_ERROR_DURATION;
    constructor(editor) {
        super(editor);
        const config = editor.config.get('aiAgent');
        this.showErrorDuration = config?.showErrorDuration ?? SHOW_ERROR_DURATION;
    }
    static get pluginName() {
        return 'AiAgentUI';
    }
    static get requires() {
        return [Widget];
    }
    /**
     * Initializes the AI Agent UI plugin, setting up UI components and event listeners.
     * This method is called when the plugin is loaded.
     */
    init() {
        try {
            const aiAgentPlugin = this.editor.plugins.get('AiAgent');
            if (!aiAgentPlugin.isEnabled) {
                return;
            }
            aiAgentContext.uiComponent = this;
            // Initialize UI components like buttons, placeholders, loaders, etc.
            this.initializeUIComponents();
            // Set displays content in the appropriate language.
            this.initializeUILanguage();
            // Attach event listeners for handling editor events and user interactions
            this.attachListener();
        }
        catch (error) {
            console.error(error.message);
        }
    }
    /**
     * Initializes UI components such as placeholders, loaders, and buttons for the editor.
     */
    initializeUIComponents() {
        const editor = this.editor;
        const t = editor.t;
        registerInlineSlashSchema(editor);
        registerAiTagSchema(editor);
        registerAiAnimateStatusSchema(editor);
        this.addGptErrorToolTip();
        addAiAgentButton(editor);
        addAiAgentToneButton(editor);
        editor.accessibility.addKeystrokeInfoGroup({
            id: 'ai-agent',
            categoryId: 'navigation',
            label: t('AI Agent'),
            keystrokes: [
                {
                    label: t('Slash Command: Open the AI Command Menu in an Empty Field'),
                    keystroke: '/'
                },
                {
                    // eslint-disable-next-line max-len
                    label: t('Force Insert Slash Command: Add a Slash Command Within Existing Text'),
                    keystroke: env.isMac ? 'Cmd + /' : 'Ctrl + /'
                },
                {
                    label: t('Cancel AI Generation'),
                    keystroke: env.isMac ? 'Cmd + Backspace' : 'Ctrl + Backspace'
                }
            ]
        });
    }
    /**
     * Initializes the UI language settings based on the editor's locale.
     * Displays an error tooltip if the current language is unsupported.
     */
    initializeUILanguage() {
        const editor = this.editor;
        const t = editor.t;
        const contentLanguageCode = editor.locale.contentLanguage;
        const supportedLanguages = SUPPORTED_LANGUAGES;
        if (!supportedLanguages.includes(contentLanguageCode)) {
            this.showGptErrorToolTip(t('Unsupported language code'));
        }
    }
    /**
     * Attaches event listeners to the editor for handling user interactions and content changes.
     */
    attachListener() {
        const editor = this.editor;
        const model = editor.model;
        model.document.selection.on('change:range', () => {
            setTimeout(() => {
                this.applyPlaceholderToCurrentLine();
            }, 10);
            const modelRoot = editor.model.document.getRoot();
            if (modelRoot) {
                const modelRange = editor.model.createRangeIn(modelRoot);
                const itemsToRemove = [];
                for (const item of modelRange.getItems()) {
                    if (item.is('element', 'inline-slash') && item.isEmpty) {
                        itemsToRemove.push(item); // Collect empty items
                    }
                }
                // Remove collected empty inline-slash elements
                editor.model.change(writer => {
                    for (const item of itemsToRemove) {
                        writer.remove(item);
                    }
                });
            }
        });
        editor.editing.view.document.on('change:isFocused', (evt, data, isFocused) => {
            if (isFocused) {
                setTimeout(() => {
                    this.applyPlaceholderToCurrentLine();
                }, 10);
            }
        });
        editor.editing.view.document.on('scroll', () => {
            this.hidePlaceHolder();
        });
        document.addEventListener('scroll', () => {
            this.hidePlaceHolder();
        });
        editor.editing.view.document.on('blur', () => {
            this.hidePlaceHolder();
        });
    }
    /**
     * Applies the placeholder to the current line in the editor if it is empty.
     * Hides the placeholder if the line is not empty.
     */
    applyPlaceholderToCurrentLine() {
        const editor = this.editor;
        const model = editor.model;
        const modelSelection = model.document.selection;
        const block = modelSelection.getFirstPosition()?.parent;
        if (block && block.isEmpty) {
            this.hidePlaceHolder();
            setTimeout(async () => {
                if (block.is('element')) {
                    const rect = await this.getRectDomOfGivenModelElement(block);
                    if (rect) {
                        this.showPlaceHolder(rect);
                    }
                }
            }, 100);
        }
        else {
            this.hidePlaceHolder();
        }
    }
    /**
     * Retrieves the DOM rectangle of a given model element.
     *
     * @param element - The model element for which to get the DOM rectangle.
     * @returns A promise that resolves to the DOMRect of the element, or null if not found.
     */
    async getRectDomOfGivenModelElement(element) {
        const editor = this.editor;
        const mapper = editor.editing.mapper;
        const view = editor.editing.view;
        const equivalentView = mapper.toViewElement(element);
        if (equivalentView) {
            const domElement = view.domConverter.mapViewToDom(equivalentView);
            if (domElement) {
                const childPos = domElement.getBoundingClientRect();
                const parentPos = editor.ui.view.editable.element?.parentElement?.getBoundingClientRect();
                const topRelative = childPos.top - (parentPos?.top ?? 0);
                const leftRelative = childPos.left - (parentPos?.left ?? 0);
                return {
                    top: topRelative,
                    left: leftRelative
                };
            }
        }
        return null;
    }
    /**
     * Adds a placeholder element to the document body for user interaction.
     */
    addPlaceholder() {
        const editor = this.editor;
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.PLACEHOLDER_TEXT_ID}`);
        if (!ele) {
            const t = editor.t;
            const placeholder = document.createElement('p');
            placeholder.id = this.PLACEHOLDER_TEXT_ID;
            placeholder.onclick = () => {
                editor.focus();
            };
            placeholder.classList.add('place-holder');
            placeholder.textContent = t('Type / to request AI content');
            const parentPanelContent = editor.ui.view.editable.element?.parentElement;
            if (parentPanelContent) {
                if (parentPanelContent.style.position !== 'absolute') {
                    parentPanelContent.style.position = 'relative';
                }
            }
            const panelContent = editor.ui.view.editable.element;
            if (panelContent) {
                panelContent.insertAdjacentElement('afterend', placeholder);
            }
        }
    }
    /**
     * Shows the placeholder at the specified position.
     *
     * @param rect - The DOMRect object defining the position to show the placeholder.
     */
    showPlaceHolder(rect) {
        this.addPlaceholder();
        const editor = this.editor;
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.PLACEHOLDER_TEXT_ID}`);
        const isReadOnlyMode = this.editor.isReadOnly;
        if (ele && rect && !isReadOnlyMode) {
            ele.classList.add('show-place-holder');
            ele.style.top = `${rect.top}px`;
            ele.style.left = `${rect.left}px`;
        }
        else if (ele) {
            ele.remove();
        }
    }
    /**
     * Hides the placeholder element from the document.
     */
    hidePlaceHolder() {
        const editor = this.editor;
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.PLACEHOLDER_TEXT_ID}`);
        if (ele) {
            ele.remove();
        }
    }
    /**
     * Adds a loader element to the document body for indicating processing.
     */
    addLoader(editor) {
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.GPT_RESPONSE_LOADER_ID}`);
        if (!ele) {
            const loaderElement = document.createElement('div');
            loaderElement.id = this.GPT_RESPONSE_LOADER_ID;
            loaderElement.classList.add('gpt-loader');
            const parentPanelContent = editor.ui.view.editable.element?.parentElement;
            if (parentPanelContent) {
                parentPanelContent.style.position = 'relative';
            }
            const panelContent = editor.ui.view.editable.element;
            if (panelContent) {
                panelContent.insertAdjacentElement('afterend', loaderElement);
            }
        }
    }
    /**
     * Shows the loader at the specified position.
     *
     * @param rect - The DOMRect object defining the position to show the loader.
     */
    showLoader(editor) {
        this.addLoader(editor);
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.GPT_RESPONSE_LOADER_ID}`);
        const domSelection = window.getSelection();
        const domRange = domSelection?.getRangeAt(0);
        const childPos = domRange.getBoundingClientRect();
        const parentPos = editor.ui.view.editable.element?.parentElement?.getBoundingClientRect();
        const top = childPos.top - (parentPos?.top ?? 0);
        const left = childPos.left - (parentPos?.left ?? 0);
        if (ele) {
            ele.style.left = `${left + 10}px`;
            ele.style.top = `${top + 10}px`;
            ele.classList.add('show-gpt-loader');
        }
    }
    /**
     * Hides the loader element from the document.
     */
    hideLoader(editor) {
        const ele = editor.ui.view.editable.element?.parentElement?.querySelector(`#${this.GPT_RESPONSE_LOADER_ID}`);
        if (ele) {
            ele.remove();
        }
    }
    /**
     * Adds an error tooltip element to the document body for displaying error messages.
     */
    addGptErrorToolTip() {
        const tooltipElement = document.createElement('p');
        tooltipElement.id = this.GPT_RESPONSE_ERROR_ID;
        tooltipElement.classList.add('response-error');
        document.body.appendChild(tooltipElement);
    }
    /**
     * Displays an error tooltip with the specified message.
     *
     * @param message - The error message to display in the tooltip.
     */
    showGptErrorToolTip(message) {
        console.log('Showing error message...', message);
        const editor = this.editor;
        const view = editor?.editing?.view?.domRoots?.get('main');
        const tooltipElement = document.getElementById(this.GPT_RESPONSE_ERROR_ID);
        const editorRect = view?.getBoundingClientRect();
        if (tooltipElement && editorRect) {
            tooltipElement.classList.add('show-response-error');
            tooltipElement.textContent = message;
            setTimeout(() => {
                this.hideGptErrorToolTip();
            }, this.showErrorDuration);
        }
    }
    /**
     * Hides the error tooltip element from the document.
     */
    hideGptErrorToolTip() {
        const tooltipElement = document.getElementById(this.GPT_RESPONSE_ERROR_ID);
        if (tooltipElement) {
            tooltipElement.classList.remove('show-response-error');
        }
    }
}
