<?php

namespace Drupal\ckeditor_custom_headings\Plugin\CKEditor5Plugin;

use Drupal\Core\Form\FormStateInterface;
use Drupal\ckeditor5\Plugin\CKEditor5Plugin\Heading as HeadingBase;
use Drupal\editor\EditorInterface;

/**
 * Provides a custom CKEditor 5 Heading plugin with configurable headings.
 */
class Heading extends HeadingBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'customize' => FALSE,
      'custom_headings' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $customize = (bool) $this->configuration['customize'];

    $form['customize'] = [
      '#type' => 'checkbox',
      '#title' => 'Customize headings',
      '#default_value' => $customize,
      '#weight' => -50,
    ];

    $form['custom_headings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Headings'),
      '#description' => $this->t('Define custom headings in the format: <code>h2.custom-heading-2|Custom heading (h2)</code>. Each line should contain a tag, an optional class prefixed with a dot, and an optional title prefixed with a pipe.'),
      '#default_value' => $this->configuration['custom_headings'],
    ];

    $form['enabled_headings']['#access'] = !$customize;
    $form['custom_headings']['#access'] = $customize;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['customize'] = $form_state->getValue('customize');
    $this->configuration['custom_headings'] = $form_state->getValue('custom_headings');
  }

  /**
   * {@inheritdoc}
   */
  public function getDynamicPluginConfig(array $static_plugin_config, EditorInterface $editor): array {
    $customHeadings = $this->configuration['custom_headings'];
    if (!$customHeadings) {
      return parent::getDynamicPluginConfig($static_plugin_config, $editor);
    }

    $items = $this->parseConfiguration($customHeadings);
    $options = [];

    foreach ($items as $item) {
      $option = [];
      $option['model'] = $this->getModelFromItem($item);

      if (!empty($item['tag']) && $item['tag'] !== 'p') {
        $option['view'] = $item['tag'];
      }

      if (!empty($item['title'])) {
        $option['title'] = $item['title'];
      }

      if (!empty($item['class'])) {
        $option['class'] = $item['class'];
        $option['view'] = ['name' => $item['tag'], 'classes' => $item['class']];
        $option['converterPriority'] = 'high';
      }

      $options[] = $option;
    }

    return [
      'heading' => [
        'options' => array_values($options),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getElementsSubset(): array {
    $customHeadings = $this->configuration['custom_headings'];
    if (!$customHeadings) {
      return parent::getElementsSubset();
    }

    $elements = $this->getPluginDefinition()->getElements();
    $items = $this->parseConfiguration($customHeadings);
    $tagsToReturn = [];

    foreach ($items as $item) {
      $tag = '<' . $item['tag'] . '>';
      if (in_array($tag, $elements)) {
        $tagsToReturn[] = $tag;
      }

      if ($item['class']) {
        $tag = '<' . $item['tag'] . ' class>';
        if (in_array($tag, $elements)) {
          $tagsToReturn[] = $tag;
        }
      }
    }

    return array_values(array_unique($tagsToReturn));
  }

  /**
   * Parses the custom headings configuration string into an array.
   */
  protected function parseConfiguration(string $enabledHeadings): array {
    $result = [];
    $lines = explode("\n", $enabledHeadings);

    foreach ($lines as $line) {
      $line = trim($line);
      if ($line === '') {
        continue;
      }

      $regexTag = '(?<tag>[^\|\.]*)';
      $regexClass = '(?:\.(?<class>[^|]*))?';
      $regexTitle = '(?:\|(?<title>.*))?';
      if (!preg_match("/^{$regexTag}{$regexClass}{$regexTitle}$/", $line, $matches)) {
        continue;
      }

      $matches = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
      $result[] = $matches;
    }

    return $result;
  }

  /**
   * Returns the model name based on the item.
   */
  protected function getModelFromItem(array $item): string {
    if ($item['tag'] === 'p') {
      $name = 'paragraph';
    }
    elseif (preg_match('/^h([1-6])$/', $item['tag'], $matches)) {
      $name = 'heading' . $matches[1];
    }
    else {
      throw new \RuntimeException('Unknown tag "' . $item['tag'] . '"');
    }

    if ($item['class']) {
      $name .= ucfirst($item['class']);
    }

    return $name;
  }

}
