<?php

declare(strict_types=1);

namespace Drupal\Tests\ckeditor_infocard\FunctionalJavascript;

use Drupal\editor\Entity\Editor;
use Drupal\filter\Entity\FilterFormat;
use Drupal\Tests\ckeditor5\FunctionalJavascript\CKEditor5TestBase;
use Drupal\Tests\ckeditor5\Traits\CKEditor5TestTrait;
use Drupal\user\RoleInterface;

/**
 * Test the CKEditor InfoCard plugin.
 *
 * Inspired by \Drupal\Tests\ckeditor5\FunctionalJavascript\CKEditor5DialogTest.
 *
 * @group ckeditor_infocard
 */
class CKEditorInfoCardTest extends CKEditor5TestBase {
  use CKEditor5TestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ckeditor_infocard',
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a filter format, and editor configuration with the InfoCard
    // plugin configured.
    FilterFormat::create([
      'format' => 'test_format',
      'name' => 'CKEditor InfoCard',
      'roles' => [RoleInterface::AUTHENTICATED_ID],
    ])->save();
    Editor::create([
      'format' => 'test_format',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => ['items' => ['sourceEditing', 'span']],
        'plugins' => ['ckeditor5_sourceEditing' => ['allowed_tags' => []]],
      ],
    ])->save();
  }

  /**
   * Test that enabling the InfoCard plugin adds 'span' to the allowed tags.
   */
  public function testInfoCardPluginAllowsSpanTag(): void {
    $testDataContent = 'Allows InfoCard Tag Test';
    $expectedOutput = '<span data-content="Web Hypertext Application Technology Working Group">WHATWG</span>';

    // Edit a page, click CKEditor button for source editing, enter text with an
    // InfoCard, go back to the regular editing mode, ensure the
    // InfoCard button is now active, save, and verify the InfoCard is
    // visible on the saved page.
    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    $this->editSourceSetContent($expectedOutput);

    $this->assertEditorButtonEnabled('InfoCard');
    $this->assertTrue($this->getEditorButton('InfoCard')->hasClass('ck-on'));
    $this->assertStringContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput);
  }

  /**
   * Test that we can edit existing InfoCards.
   */
  public function testEditInfoCard(): void {
    $testDataContent = 'Edit InfoCard';
    $expectedOutput1 = '<span data-content="Math Working Group">MWG</span>';
    $expectedOutput2 = '<span data-content="Media Working Group">MWG</span>';

    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);
    $this->waitForEditor();
    $this->editSourceSetContent($expectedOutput1);
    $this->assertStringContainsString($expectedOutput1, $this->getEditorDataAsHtmlString());
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput1);

    $node = $this->drupalGetNodeByTitle($testDataContent);
    $this->drupalGet($node->toUrl('edit-form'));
    $this->waitForEditor();
    $this->assertStringContainsString($expectedOutput1, $this->getEditorDataAsHtmlString());

    // Change the InfoCard, and make sure it is saved.
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');
    [, $infoCardHintText, $infoCardDataContent, , $saveButton] = $this->checkInfoCardBalloon();
    $this->assertStringContainsString('MWG', $infoCardHintText->getValue());
    $infoCardDataContent->setValue('Media Working Group');
    $saveButton->click();
    $this->assertStringContainsString($expectedOutput2, $this->getEditorDataAsHtmlString());
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been updated.");
    $this->assertSession()->responseContains($expectedOutput2);
  }

  /**
   * Test that we can add an InfoCard by selecting the InfoCard text.
   */
  public function testExistingTextActuallyAdd(): void {
    $testDataContent = 'Actually Add InfoCard To Existing Text';
    $expectedOutput = '<span data-content="Accessible Rich Internet Applications Working Group">ARIAWG</span>';

    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    $this->waitForEditor();
    $this->editSourceSetContent('The mission of the ARIAWG is to enhance the accessibility of web content.');

    $this->selectTextInsideEditor('ARIAWG');
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    [, $infoCardHintText, $infoCardDataContent, , $saveButton] = $this->checkInfoCardBalloon();
    $this->assertStringContainsString('ARIAWG', $infoCardHintText->getValue());
    $infoCardDataContent->setValue('Accessible Rich Internet Applications Working Group');
    $saveButton->click();

    // Make sure the editor text contains the InfoCard we just created.
    $this->assertStringContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response contains the InfoCard we
    // just created.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()
      ->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput);
  }

  /**
   * Test that clicking the cancel button works correctly with text selected.
   */
  public function testExistingTextCancelAdd() {
    $testDataContent = 'Cancel Adding InfoCard To Existing Text';
    $expectedOutput = '<span data-content="Authoring Tool Accessibility Guidelines Working Group">ATAGWG</span>';

    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    $this->waitForEditor();
    $this->editSourceSetContent('The ATAGWG was chartered to maintain and support the Authoring Tool Accessibility Guidelines.');

    $this->selectTextInsideEditor('ATAGWG');
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    [, $infoCardHintText, $infoCardDataContent, $cancelButton] = $this->checkInfoCardBalloon();
    $this->assertStringContainsString('ATAGWG', $infoCardHintText->getValue());
    $infoCardDataContent->setValue('Authoring Tool Accessibility Guidelines Working Group');
    $cancelButton->click();

    // Make sure the editor text does not contain the InfoCard we just
    // cancelled.
    $this->assertStringNotContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response does not contain the
    // InfoCard we just cancelled.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()
      ->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseNotContains($expectedOutput);
  }

  /**
   * Test InfoCard with no DataContent through text selection.
   */
  public function testExistingTextNoDataContent(): void {
    $testDataContent = 'Add InfoCard With No DataContent To Existing Text';
    $expectedOutput = '<span>JSON-LD</span>';

    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    $this->waitForEditor();
    $this->editSourceSetContent('The mission of the JSON-LD Working Group is to maintain the family of Recommendations and related Working Group Notes.');

    $this->selectTextInsideEditor('JSON-LD');
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    [, $infoCardHintText, $infoCardDataContent, , $saveButton] = $this->checkInfoCardBalloon();
    $this->assertStringContainsString('JSON-LD', $infoCardHintText->getValue());
    $infoCardDataContent->setValue('');
    $saveButton->click();

    // Make sure the editor text contains the infoCard we just created.
    $this->assertStringContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response contains the InfoCard we
    // just created.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()
      ->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput);
  }

  /**
   * Test that we can add an InfoCard without selecting text.
   */
  public function testNewInfoCardActuallyAdd(): void {
    $testDataContent = 'Actually Add New InfoCard Test';
    $expectedOutput = '<span data-content="World Wide Web Consortium">W3C</span>';

    // Load a node/edit page and set a title.
    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    // Try adding an InfoCard, i.e.: without selecting text first.
    $this->waitForEditor();
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    // Check the InfoCard balloon, then
    // set InfoCard HintText and DataContent, and
    // click the balloon's Save button.
    [, $infoCardHintText, $infoCardDataContent, , $saveButton] = $this->checkInfoCardBalloon();
    $infoCardHintText->setValue('W3C');
    $infoCardDataContent->setValue('World Wide Web Consortium');
    $saveButton->click();

    // Make sure the editor text contains the InfoCard we just created.
    $this->assertStringContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response contains the InfoCard we
    // just created.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput);
  }

  /**
   * Test that clicking the cancel button works correctly without text selected.
   */
  public function testNewInfoCardCancelAdd(): void {
    $testDataContent = 'Cancel Add New InfoCard Test';
    $expectedOutput = '<span data-content="Web Applications Working Group">WAWG</span>';

    // Load a node/edit page and set a title.
    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    // Try adding an InfoCard, i.e.: without selecting text first.
    $this->waitForEditor();
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    // Check the InfoCard balloon, then set InfoCard Hint and DataContent, and
    // click the balloon's Cancel button.
    [, $infoCardHintText, $infoCardDataContent, $cancelButton] = $this->checkInfoCardBalloon();
    $infoCardHintText->setValue('WAWG');
    $infoCardDataContent->setValue('Web Applications Working Group');
    $cancelButton->click();

    // Make sure the editor text does not contain the InfoCard we just
    // cancelled.
    $this->assertStringNotContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response does not contain the
    // InfoCard we just cancelled.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseNotContains($expectedOutput);
  }

  /**
   * Test an InfoCard with no DataContent, without selecting text.
   */
  public function testNewInfoCardNoDataContent(): void {
    $testDataContent = 'Add New InfoCard With No DataContent';
    $expectedOutput = '<span>PNG</span>';

    $this->drupalGet('node/add');
    $this->getSession()->getPage()
      ->fillField('data-content[0][value]', $testDataContent);

    // Try adding an InfoCard, i.e.: without selecting text first.
    $this->waitForEditor();
    $this->assertEditorButtonEnabled('InfoCard');
    $this->pressEditorButton('InfoCard');

    [, $infoCardHintText, $infoCardDataContent, , $saveButton] = $this->checkInfoCardBalloon();
    $infoCardHintText->setValue('PNG');
    $infoCardDataContent->setValue('');
    $saveButton->click();

    // Make sure the editor text contains the InfoCard we just created.
    $this->assertStringContainsString($expectedOutput, $this->getEditorDataAsHtmlString());

    // Save the page and make sure the response contains the InfoCard we
    // just created.
    $this->getSession()->getPage()
      ->pressButton('Save');
    $this->assertSession()->statusMessageContains("page $testDataContent has been created.");
    $this->assertSession()->responseContains($expectedOutput);
  }

  /**
   * Check that the InfoCard prompt balloon appears with correct controls.
   *
   * @return \Behat\Mink\Element\NodeElement[]
   *   An array of five NodeElements in the following order, suitable for Array
   *   Unpacking / Array Destructuring (i.e.: with list() or [...]):
   *   1. the Balloon containing all the controls;
   *   2. the InfoCard Hint Text field;
   *   3. the InfoCard DataContent field;
   *   4. the Balloon's Cancel button; and;
   *   5. the Balloon's Save button.
   */
  protected function checkInfoCardBalloon(): array {
    $balloon = $this->assertVisibleBalloon('-span-form');

    $infoCardText = $balloon->findField('Add InfoCard');
    $this->assertNotEmpty($infoCardText);

    $infoCardDataContent = $balloon->findField('Add text');
    $this->assertNotEmpty($infoCardDataContent);

    $cancelButton = $this->getBalloonButton('Cancel');
    $this->assertNotEmpty($cancelButton);

    $saveButton = $this->getBalloonButton('Save');
    $this->assertNotEmpty($saveButton);

    return [$balloon, $infoCardText, $infoCardDataContent, $cancelButton, $saveButton];
  }

  /**
   * Edit a CKEditor field's source, setting it to the given content.
   *
   * @param string $content
   *   The content to place in the CKEditor field.
   */
  protected function editSourceSetContent(string $content): void {
    $this->waitForEditor();
    $this->assertEditorButtonEnabled('Source');
    $this->pressEditorButton('Source');
    $this->assertEditorButtonDisabled('InfoCard');
    $sourceTextArea = $this->assertSession()
      ->waitForElement('css', '.ck-source-editing-area textarea');
    $sourceTextArea->setValue($content);
    $this->assertEditorButtonEnabled('Source');
    $this->pressEditorButton('Source');
    $this->assertEditorButtonEnabled('InfoCard');
  }

  /**
   * Find text inside the CKEditor, and select it.
   *
   * @param string $textToSelect
   *   The text to find and select.
   */
  protected function selectTextInsideEditor(string $textToSelect): void {
    $javascript = <<<JS
(function() {
  const textToSelect = "$textToSelect";
  const el = document.querySelector(".ck-editor__main .ck-editor__editable p");
  const startIndex = el.textContent.indexOf(textToSelect);
  const endIndex = startIndex + textToSelect.length;
  const range = document.createRange();
  const sel = window.getSelection();

  sel.removeAllRanges();
  range.setStart(el.firstChild, startIndex);
  range.setEnd(el.firstChild, endIndex);
  sel.addRange(range);
})();
JS;
    $this->getSession()->evaluateScript($javascript);
  }

}
