<?php

declare(strict_types=1);

namespace Drupal\ckeditor_media_edit\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Redirect to the media edit route by UUID.
 */
final class EditMediaModalController extends ControllerBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('entity.repository'));
  }

  /**
   * Constructs a new EditMediaModalController object.
   */
  public function __construct(protected readonly EntityRepositoryInterface $entityRepository) {}

  /**
   * Access callback.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   * @param string $uuid
   *   The media uuid.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, string $uuid): AccessResultInterface {
    $entity = $this->entityRepository->loadEntityByUuid('media', $uuid);

    if (empty($entity)) {
      return AccessResult::forbidden();
    }

    return $entity->access('update', $account, TRUE);
  }

  /**
   * Get the edit media url by the UUID.
   *
   * At the time where the CKEditor5 plugin wants to open the edit modal it
   * only has access to the UUID of the media entity, not the ID.
   *
   * @param string $uuid
   *   The media uuid.
   * @param string $form_mode
   *   The form mode.
   *
   * @return array
   *   Entity form renderable.
   */
  public function getEditForm(string $uuid, string $form_mode): array {
    $entity = $this->entityRepository->loadEntityByUuid('media', $uuid);
    \assert($entity instanceof MediaInterface);
    $form = $this->entityFormBuilder()->getForm($entity, $form_mode, [
      'ckeditor_media_edit' => TRUE,
    ]);
    return $form;
  }

}
