<?php

declare(strict_types=1);

namespace Drupal\Tests\ckeditor_media_edit\FunctionalJavascript;

use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\Entity\EntityViewMode;
use Drupal\editor\Entity\Editor;
use Drupal\file\Entity\File;
use Drupal\filter\Entity\FilterFormat;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\node\NodeInterface;
use Drupal\Tests\ckeditor5\Traits\CKEditor5TestTrait;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\user\UserInterface;

/**
 * Functional JS test class for the module.
 *
 * @group ckeditor_media_edit
 */
final class EditMediaModalTest extends WebDriverTestBase {

  use CKEditor5TestTrait;
  use MediaTypeCreationTrait;
  use TestFileCreationTrait;

  /**
   * The user to use during testing.
   */
  protected UserInterface $adminUser;

  /**
   * The sample Media entity to embed.
   */
  protected MediaInterface $media;

  /**
   * A host entity with a body field to embed media in.
   */
  protected NodeInterface $host;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ckeditor5',
    'media',
    'node',
    'text',
    'media_test_embed',
    'media_library',
    'ckeditor5_test',
    'ckeditor_media_edit',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'starterkit_theme';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    EntityViewMode::create([
      'id' => 'media.view_mode_1',
      'targetEntityType' => 'media',
      'status' => TRUE,
      'enabled' => TRUE,
      'label' => 'View Mode 1',
    ])->save();

    FilterFormat::create([
      'format' => 'test_format',
      'name' => 'Test format',
      'filters' => [
        'filter_html' => [
          'status' => TRUE,
          'settings' => [
            'allowed_html' => '<p> <br> <strong> <em> <a href> <drupal-media data-entity-type data-entity-uuid data-align data-view-mode data-caption alt>',
          ],
        ],
        'filter_align' => ['status' => TRUE],
        'filter_caption' => ['status' => TRUE],
        'media_embed' => [
          'status' => TRUE,
          'settings' => [
            'default_view_mode' => 'view_mode_1',
            'allowed_view_modes' => ['view_mode_1'],
            'allowed_media_types' => [],
          ],
        ],
      ],
    ])->save();

    Editor::create([
      'editor' => 'ckeditor5',
      'format' => 'test_format',
      'settings' => [
        'toolbar' => [
          'items' => [
            'sourceEditing',
            'link',
            'bold',
            'italic',
            'drupalMedia',
          ],
        ],
        'plugins' => [
          'ckeditor5_sourceEditing' => [
            'allowed_tags' => [],
          ],
          'media_media' => [
            'allow_view_mode_override' => TRUE,
          ],
        ],
      ],
      'image_upload' => [
        'status' => FALSE,
      ],
    ])->save();

    // Note that media_install() grants 'view media' to all users by default.
    $user = $this->drupalCreateUser([
      'use text format test_format',
      'bypass node access',
      'administer media',
    ]);
    \assert($user !== FALSE);
    $this->adminUser = $user;

    // Create a sample media entity to be embedded.
    $this->createMediaType('image', ['id' => 'image']);
    $file = $this->getTestFiles('image')[0];
    File::create([
      // @phpstan-ignore-next-line
      'uri' => $file->uri,
    ])->save();
    $this->media = Media::create([
      'bundle' => 'image',
      'name' => 'Screaming hairy armadillo',
      'field_media_image' => [
        [
          'target_id' => 1,
          'alt' => 'default alt',
          'title' => 'default title',
        ],
      ],
    ]);
    $this->media->setOwner($this->adminUser);
    $this->media->save();

    // Set created media types for each view mode.
    EntityViewDisplay::create([
      'id' => 'media.image.view_mode_1',
      'targetEntityType' => 'media',
      'status' => TRUE,
      'bundle' => 'image',
      'mode' => 'view_mode_1',
    ])
      ->setComponent('name', ['type' => 'text_default'])
      ->save();

    $entity_form_display = EntityFormDisplay::load('media.image.default');
    \assert($entity_form_display !== NULL);
    $entity_form_display->setComponent('name', ['type' => 'string_textfield']);

    // Create a sample host entity to embed media in.
    $this->drupalCreateContentType(['type' => 'blog']);
    $this->host = $this->createNode([
      'type' => 'blog',
      'title' => 'Animals with strange names',
      'body' => [
        'value' => '<drupal-media data-entity-type="media" data-entity-uuid="' . $this->media->uuid() . '"></drupal-media>',
        'format' => 'test_format',
      ],
    ]);
    $this->host->save();

    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test that media can be updated from CKEditor5.
   */
  public function testUpdate(): void {
    $this->drupalGet($this->host->toUrl('edit-form'));
    $this->waitForEditor();
    /** @var \Drupal\FunctionalJavascriptTests\JSWebAssert */
    $assert_session = $this->assertSession();

    // Name is present in the previewed media.
    self::assertNotEmpty($preview = $assert_session->waitForElementVisible('css', '.ck-widget.drupal-media > [data-drupal-media-preview="ready"] > .media', 30000));
    self::assertStringContainsString('Screaming hairy armadillo', $preview->getHtml());

    // Test that clicking the media widget triggers a CKEditor balloon panel
    // with an "Edit media" button.
    $this->click('.ck-widget.drupal-media');
    $this->assertVisibleBalloon('[aria-label="Drupal Media toolbar"]');

    $this->getBalloonButton('Edit media')->click();

    // Update the media name.
    $modal = $assert_session->waitForElementVisible('css', '[role="dialog"].edit-media-modal');
    self::assertNotEmpty($modal);
    $name_field = $modal->find('css', 'input[name="name[0][value]"]');
    $new_name = 'Savory Crisp Airway';
    $name_field->setValue($new_name);
    $assert_session->elementExists('css', '.ui-dialog-buttonpane')->pressButton('Save');
    $assert_session->assertWaitOnAjaxRequest();

    // Assert media has been updated.
    $media_storage = \Drupal::entityTypeManager()->getStorage('media');
    $media_storage->resetCache([$this->getMediaId()]);
    $this->media = $this->loadMedia($this->getMediaId());
    self::assertEquals($new_name, $this->media->label());

    // Assert media has been updated in the CKEditor preview area.
    $assert_session->waitForElementRemoved('css', '[role="dialog"].edit-media-modal');
    \usleep(10000);
    $ckeditor_name_field = $this->getSession()->getPage()->find('css', '.ck-editor__main .field--name-name .field__item');
    self::assertEquals($new_name, $ckeditor_name_field->getText());
  }

  /**
   * Helper media loader.
   */
  private function loadMedia(string $id): MediaInterface {
    $media = Media::load($id);
    if (!$media instanceof MediaInterface) {
      throw new \Exception('Media  does not exist.');
    }
    return $media;
  }

  /**
   * Get media ID.
   */
  private function getMediaId(): string {
    return (string) $this->media->id();
  }

}
