/**
 * @file
 * CKEditor 5 Media Image Style plugin for Drupal.
 */

(function (Drupal, CKEditor5) {
  'use strict';

  var Plugin = CKEditor5.core.Plugin;
  var Command = CKEditor5.core.Command;

  var ButtonView = CKEditor5.ui.ButtonView;
  var ContextualBalloon = CKEditor5.ui.ContextualBalloon;
  var View = CKEditor5.ui.View;
  var LabeledFieldView = CKEditor5.ui.LabeledFieldView;
  var createLabeledInputText = CKEditor5.ui.createLabeledInputText;
  var submitHandler = CKEditor5.ui.submitHandler;
  var clickOutsideHandler = CKEditor5.ui.clickOutsideHandler;

  var FocusTracker = CKEditor5.utils.FocusTracker;
  var KeystrokeHandler = CKEditor5.utils.KeystrokeHandler;

  var checkIcon = '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M6.972 16.615a.997.997 0 0 1-.744-.292l-4.596-4.596a1 1 0 1 1 1.414-1.414l3.926 3.926 9.937-9.937a1 1 0 0 1 1.414 1.415L7.717 16.323a.997.997 0 0 1-.745.292z"/></svg>';

  var cancelIcon = '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="m11.591 10.177 4.243 4.242a1 1 0 0 1-1.415 1.415l-4.242-4.243-4.243 4.243a1 1 0 0 1-1.414-1.415l4.243-4.242L4.52 5.934A1 1 0 0 1 5.934 4.52l4.243 4.243 4.242-4.243a1 1 0 1 1 1.415 1.414l-4.243 4.243z"/></svg>';

  var imageStyleIcon = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><path d="M4 2h4v2H4v4H2V4a2 2 0 0 1 2-2zm12 0a2 2 0 0 1 2 2v4h-2V4h-4V2h4zM4 18v-4H2v4a2 2 0 0 0 2 2h4v-2H4zm12 0h-4v2h4a2 2 0 0 0 2-2v-4h-2v4zM8 8h4v4H8z"/></svg>';
  
  // Inject dynamic CSS for image style dimensions
  function injectStyleCSS(dimensions) {
    var styleId = 'ckeditor5-media-image-style-css';
    var existingStyle = document.getElementById(styleId);
    
    if (existingStyle) {
      existingStyle.remove();
    }

    if (!dimensions || Object.keys(dimensions).length === 0) {
      return;
    }

    var css = '';
    
    Object.keys(dimensions).forEach(function(styleId) {
      var dim = dimensions[styleId];
      var width = dim.width ? dim.width + 'px' : 'auto';
      var height = dim.height ? dim.height + 'px' : 'auto';
      
      css += 'figure.drupal-media[data-image-style="' + styleId + '"] img {\n';
      if (dim.width) {
        css += '  max-width: ' + width + ' !important;\n';
        css += '  width: ' + width + ' !important;\n';
      }
      if (dim.height) {
        css += '  max-height: ' + height + ' !important;\n';
        css += '  height: ' + (dim.width ? 'auto' : height) + ' !important;\n';
      }
      css += '  object-fit: contain;\n';
      css += '}\n\n';
    });

    var styleElement = document.createElement('style');
    styleElement.id = styleId;
    styleElement.textContent = css;
    document.head.appendChild(styleElement);
  }

  class MediaImageStyleCommand extends Command {
    refresh() {
      var element = this._getSelectedMedia();
      this.isEnabled = !!element;

      if (this.isEnabled) {
        this.value = element.getAttribute('drupalMediaImageStyle') || '';
      } else {
        this.value = '';
      }
    }

    execute(options) {
      var model = this.editor.model;
      var element = this._getSelectedMedia();

      if (!element) {
        return;
      }

      model.change(function(writer) {
        if (options.newValue) {
          writer.setAttribute('drupalMediaImageStyle', options.newValue, element);
        } else {
          writer.removeAttribute('drupalMediaImageStyle', element);
        }
      });
    }

    _getSelectedMedia() {
      var model = this.editor.model;
      var selection = model.document.selection;
      var selectedElement = selection.getSelectedElement();

      if (selectedElement && selectedElement.is('element', 'drupalMedia')) {
        return selectedElement;
      }

      return null;
    }
  }

  class MediaImageStyleFormView extends View {
    constructor(locale, styles) {
      super(locale);

      var t = locale.t;

      this.styles = styles || {};
      this.focusTracker = new FocusTracker();
      this.keystrokes = new KeystrokeHandler();

      this.labeledInput = new LabeledFieldView(locale, createLabeledInputText);
      this.labeledInput.label = t('Image style');

      this.saveButtonView = new ButtonView(locale);
      this.saveButtonView.set({
        label: t('Save'),
        icon: checkIcon,
        tooltip: true,
        withText: false,
        class: 'ck-button-save'
      });
      this.saveButtonView.type = 'submit';

      this.cancelButtonView = new ButtonView(locale);
      this.cancelButtonView.set({
        label: t('Cancel'),
        icon: cancelIcon,
        tooltip: true,
        withText: false,
        class: 'ck-button-cancel'
      });
      this.cancelButtonView.delegate('execute').to(this, 'cancel');

      this.setTemplate({
        tag: 'form',
        attributes: {
          class: ['ck', 'ck-media-alternative-text-form', 'ck-vertical-form', 'ck-media-image-style-form'],
          tabindex: '-1'
        },
        children: [
          this.labeledInput,
          this.saveButtonView,
          this.cancelButtonView
        ]
      });
    }

    render() {
      super.render();

      var self = this;

      var inputWrapper = this.labeledInput.fieldView.element.parentNode;
      var inputElement = this.labeledInput.fieldView.element;
      
      var selectElement = document.createElement('select');
      selectElement.className = 'ck ck-input ck-input-text ck-media-image-style-select';
      selectElement.id = inputElement.id;
      
      if (inputElement.getAttribute('aria-describedby')) {
        selectElement.setAttribute('aria-describedby', inputElement.getAttribute('aria-describedby'));
      }

      Object.keys(this.styles).forEach(function(styleId) {
        var option = document.createElement('option');
        option.value = styleId;
        option.textContent = self.styles[styleId];
        selectElement.appendChild(option);
      });

      inputWrapper.replaceChild(selectElement, inputElement);
      this._selectElement = selectElement;

      this.labeledInput.element.classList.remove('ck-labeled-field-view_empty');

      submitHandler({ view: this });

      this.focusTracker.add(this._selectElement);
      this.focusTracker.add(this.saveButtonView.element);
      this.focusTracker.add(this.cancelButtonView.element);

      this.keystrokes.listenTo(this.element);
    }

    destroy() {
      super.destroy();
      this.focusTracker.destroy();
      this.keystrokes.destroy();
    }

    focus() {
      if (this._selectElement) {
        this._selectElement.focus();
      }
    }

    getValue() {
      return this._selectElement ? this._selectElement.value : '';
    }

    setValue(value) {
      if (this._selectElement) {
        this._selectElement.value = value || '';
      }
    }
  }

  class MediaImageStyleEditing extends Plugin {
    static get pluginName() {
      return 'MediaImageStyleEditing';
    }

    init() {
      var editor = this.editor;
      var schema = editor.model.schema;

      if (schema.isRegistered('drupalMedia')) {
        schema.extend('drupalMedia', {
          allowAttributes: ['drupalMediaImageStyle']
        });
      }

      this._setupConversion();
      editor.commands.add('mediaImageStyle', new MediaImageStyleCommand(editor));

      // Inject CSS for dimensions
      var config = editor.config.get('mediaImageStyle');
      if (config && config.dimensions) {
        injectStyleCSS(config.dimensions);
      }
    }

    _setupConversion() {
      var editor = this.editor;
      var conversion = editor.conversion;

      conversion.for('upcast').attributeToAttribute({
        view: {
          name: 'drupal-media',
          key: 'data-image-style'
        },
        model: 'drupalMediaImageStyle'
      });

      conversion.for('dataDowncast').attributeToAttribute({
        model: 'drupalMediaImageStyle',
        view: 'data-image-style'
      });

      conversion.for('editingDowncast').attributeToAttribute({
        model: 'drupalMediaImageStyle',
        view: 'data-image-style'
      });
    }
  }

  class MediaImageStyleUI extends Plugin {
    static get requires() {
      return [ContextualBalloon];
    }

    static get pluginName() {
      return 'MediaImageStyleUI';
    }

    init() {
      var editor = this.editor;
      var self = this;

      this._balloon = editor.plugins.get(ContextualBalloon);
      this._form = null;

      editor.ui.componentFactory.add('mediaImageStyle', function(locale) {
        var command = editor.commands.get('mediaImageStyle');
        var button = new ButtonView(locale);

        button.set({
          label: Drupal.t('Media image style'),
          icon: imageStyleIcon,
          tooltip: true
        });

        button.bind('isEnabled').to(command, 'isEnabled');

        button.on('execute', function() {
          self._showForm();
        });

        return button;
      });
    }

    destroy() {
      super.destroy();

      if (this._form) {
        this._form.destroy();
      }
    }

    _getStyles() {
      var config = this.editor.config.get('mediaImageStyle');
      return config && config.styles ? config.styles : {};
    }

    _createForm() {
      var editor = this.editor;
      var form = new MediaImageStyleFormView(editor.locale, this._getStyles());
      var self = this;

      this.listenTo(form, 'submit', function() {
        var value = form.getValue();
        editor.execute('mediaImageStyle', { newValue: value });
        self._hideForm();
      });

      this.listenTo(form, 'cancel', function() {
        self._hideForm();
      });

      clickOutsideHandler({
        emitter: form,
        activator: function() {
          return self._isVisible;
        },
        contextElements: function() {
          return [self._balloon.view.element];
        },
        callback: function() {
          self._hideForm();
        }
      });

      form.keystrokes.set('Esc', function(data, cancel) {
        self._hideForm();
        cancel();
      });

      return form;
    }

    _showForm() {
      if (this._isVisible) {
        return;
      }

      if (!this._form) {
        this._form = this._createForm();
      }

      var command = this.editor.commands.get('mediaImageStyle');

      this._balloon.add({
        view: this._form,
        position: this._getBalloonPositionData()
      });

      this._form.setValue(command.value);
      this._form.focus();
    }

    _hideForm() {
      if (!this._isVisible) {
        return;
      }

      this._form.setValue('');
      this._balloon.remove(this._form);
      this.editor.editing.view.focus();
    }

    get _isVisible() {
      return this._form && this._balloon.visibleView === this._form;
    }

    _getBalloonPositionData() {
      var view = this.editor.editing.view;
      var viewDocument = view.document;

      return {
        target: function() {
          var selection = viewDocument.selection;
          var selectedElement = selection.getSelectedElement();

          if (selectedElement) {
            return view.domConverter.mapViewToDom(selectedElement);
          }

          return view.domConverter.viewRangeToDom(selection.getFirstRange());
        }
      };
    }
  }

  class MediaImageStyle extends Plugin {
    static get requires() {
      return [MediaImageStyleEditing, MediaImageStyleUI];
    }

    static get pluginName() {
      return 'MediaImageStyle';
    }
  }

  CKEditor5.mediaImageStyle = {
    MediaImageStyle: MediaImageStyle
  };
})(Drupal, CKEditor5);