<?php

namespace Drupal\ckeditor_media_image_style\Plugin\CKEditor5Plugin;

use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableInterface;
use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableTrait;
use Drupal\ckeditor5\Plugin\CKEditor5PluginDefault;
use Drupal\ckeditor5\Plugin\CKEditor5PluginElementsSubsetInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\editor\EditorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CKEditor 5 Media Image Style plugin.
 */
class MediaImageStyle extends CKEditor5PluginDefault implements CKEditor5PluginConfigurableInterface, CKEditor5PluginElementsSubsetInterface, ContainerFactoryPluginInterface {

  use CKEditor5PluginConfigurableTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a MediaImageStyle object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'enabled' => FALSE,
      'image_styles' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable media image style selector'),
      '#description' => $this->t('When enabled, a button will appear in the media toolbar to select the image style for embedded media images.'),
      '#default_value' => $this->configuration['enabled'],
    ];

    $image_styles = $this->getImageStyleOptions();
    
    $form['image_styles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Available image styles'),
      '#description' => $this->t('Select which image styles will be available in the editor. If none are selected, all image styles will be available.'),
      '#options' => $image_styles,
      '#default_value' => $this->configuration['image_styles'] ?: [],
      '#states' => [
        'visible' => [
          ':input[name="editor[settings][plugins][ckeditor_media_image_style_mediaImageStyle][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['enabled'] = (bool) $form_state->getValue('enabled');
    $image_styles = array_filter($form_state->getValue('image_styles'));
    $this->configuration['image_styles'] = array_values($image_styles);
  }

  /**
   * {@inheritdoc}
   */
  public function getDynamicPluginConfig(array $static_plugin_config, EditorInterface $editor): array {
    if (!$this->configuration['enabled']) {
      return [];
    }

    $configured_styles = $this->configuration['image_styles'];
    $all_styles = $this->getImageStyleOptions();
    
    if (empty($configured_styles)) {
      $styles_to_use = array_keys($all_styles);
    }
    else {
      $styles_to_use = $configured_styles;
    }

    $image_style_options = [
      '' => $this->t('Original (no style)'),
    ];
    
    $style_dimensions = [];
    
    foreach ($styles_to_use as $style_id) {
      if (isset($all_styles[$style_id])) {
        $image_style_options[$style_id] = $all_styles[$style_id];
        
        // Get dimensions for this style
        $dimensions = $this->getImageStyleDimensions($style_id);
        if ($dimensions) {
          $style_dimensions[$style_id] = $dimensions;
        }
      }
    }

    return [
      'drupalMedia' => [
        'toolbar' => ['mediaImageStyle'],
      ],
      'mediaImageStyle' => [
        'styles' => $image_style_options,
        'dimensions' => $style_dimensions,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getElementsSubset(): array {
    if ($this->configuration['enabled']) {
      return ['<drupal-media data-image-style>'];
    }
    return [];
  }

  /**
   * Get all available image style options.
   */
  protected function getImageStyleOptions(): array {
    $options = [];
    $image_styles = $this->entityTypeManager->getStorage('image_style')->loadMultiple();
    
    foreach ($image_styles as $style) {
      $options[$style->id()] = $style->label();
    }
    
    return $options;
  }

  /**
   * Get dimensions for an image style.
   */
  protected function getImageStyleDimensions(string $style_id): ?array {
    $style = $this->entityTypeManager->getStorage('image_style')->load($style_id);
    
    if (!$style) {
      return NULL;
    }

    $effects = $style->getEffects();
    
    foreach ($effects as $effect) {
      $config = $effect->getConfiguration();
      $effect_id = $config['id'] ?? '';
      
      // Check for scale, resize, or crop effects
      if (in_array($effect_id, ['image_scale', 'image_scale_and_crop', 'image_resize', 'image_crop'])) {
        $data = $config['data'] ?? [];
        $width = $data['width'] ?? NULL;
        $height = $data['height'] ?? NULL;
        
        if ($width || $height) {
          return [
            'width' => $width,
            'height' => $height,
          ];
        }
      }
    }

    return NULL;
  }

}