<?php

namespace Drupal\ckeditor_media_modal_edit;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;

/**
 * Provides modal edit links for media entities.
 */
class MediaEditModalLinkBuilder {

  /**
   * The entity type manager service.
   *
   * Used to load media entities and access storage handlers for entity types.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The renderer service.
   *
   * Used to render edit link.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  public function __construct(EntityTypeManagerInterface $entityTypeManager, RendererInterface $renderer) {
    $this->entityTypeManager = $entityTypeManager;
    $this->renderer = $renderer;
  }

  /**
   * Adds modal edit links to a media_library_select_form array.
   *
   * @param array &$form
   *   The form structure to modify.
   */
  public function addEditLinksToMediaForm(array &$form): void {
    if (!isset($form['media_library_select_form']) || !is_array($form['media_library_select_form'])) {
      return;
    }

    foreach ($form['media_library_select_form'] as &$entity) {
      if (isset($entity['#return_value']) && is_numeric($entity['#return_value'])) {
        $mid = $entity['#return_value'];
        $media = $this->entityTypeManager->getStorage('media')->load($mid);

        if ($media && $media->access('update')) {
          $url = Url::fromRoute('entity.media.edit_form', ['media' => $mid], [
            'attributes' => [
              'class' => ['use-ajax media-library-item__edit icon-link'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => json_encode(['width' => 800]),
            ],
          ]);

          $link = Link::fromTextAndUrl('Edit this media', $url)->toString();
          $entity['#suffix'] = '<span class="media-edit-link">' . $link . '</span>';
        }
      }

      // Rework styling so edit link appears inline pencil icon.
      $form['#attached']['library'][] = 'ckeditor_media_modal_edit/media-edit-links';
    }
  }

}
