/**
 * @file
 * CKEditor 5 Media Title plugin for Drupal.
 */

(function (Drupal, CKEditor5) {
  'use strict';

  var Plugin = CKEditor5.core.Plugin;
  var Command = CKEditor5.core.Command;

  var ButtonView = CKEditor5.ui.ButtonView;
  var ContextualBalloon = CKEditor5.ui.ContextualBalloon;
  var View = CKEditor5.ui.View;
  var LabeledFieldView = CKEditor5.ui.LabeledFieldView;
  var createLabeledInputText = CKEditor5.ui.createLabeledInputText;
  var submitHandler = CKEditor5.ui.submitHandler;
  var clickOutsideHandler = CKEditor5.ui.clickOutsideHandler;

  var FocusTracker = CKEditor5.utils.FocusTracker;
  var KeystrokeHandler = CKEditor5.utils.KeystrokeHandler;

  // SVG icons
  var checkIcon = '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M6.972 16.615a.997.997 0 0 1-.744-.292l-4.596-4.596a1 1 0 1 1 1.414-1.414l3.926 3.926 9.937-9.937a1 1 0 0 1 1.414 1.415L7.717 16.323a.997.997 0 0 1-.745.292z"/></svg>';

  var cancelIcon = '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="m11.591 10.177 4.243 4.242a1 1 0 0 1-1.415 1.415l-4.242-4.243-4.243 4.243a1 1 0 0 1-1.414-1.415l4.243-4.242L4.52 5.934A1 1 0 0 1 5.934 4.52l4.243 4.243 4.242-4.243a1 1 0 1 1 1.415 1.414l-4.243 4.243z"/></svg>';

  var titleIcon = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><path d="M3 4.5A1.5 1.5 0 0 1 4.5 3h11A1.5 1.5 0 0 1 17 4.5v1a.5.5 0 0 1-1 0V5H10.5v10H12a.5.5 0 0 1 0 1H8a.5.5 0 0 1 0-1h1.5V5H4v.5a.5.5 0 0 1-1 0v-1Z"/></svg>';

  class MediaImageTitleCommand extends Command {
    refresh() {
      var element = this._getSelectedMedia();
      this.isEnabled = !!element;

      if (this.isEnabled) {
        this.value = element.getAttribute('drupalMediaEntityTitle') || '';
      } else {
        this.value = '';
      }
    }

    execute(options) {
      var model = this.editor.model;
      var element = this._getSelectedMedia();

      if (!element) {
        return;
      }

      model.change(function(writer) {
        if (options.newValue) {
          writer.setAttribute('drupalMediaEntityTitle', options.newValue, element);
        } else {
          writer.removeAttribute('drupalMediaEntityTitle', element);
        }
      });
    }

    _getSelectedMedia() {
      var model = this.editor.model;
      var selection = model.document.selection;
      var selectedElement = selection.getSelectedElement();

      if (selectedElement && selectedElement.is('element', 'drupalMedia')) {
        return selectedElement;
      }

      return null;
    }
  }

  class MediaTitleFormView extends View {
    constructor(locale) {
      super(locale);

      var t = locale.t;

      this.focusTracker = new FocusTracker();
      this.keystrokes = new KeystrokeHandler();

      this.labeledInput = new LabeledFieldView(locale, createLabeledInputText);
      this.labeledInput.label = t('Title text override');
      this.labeledInput.infoText = t('Leave blank to use the default title.');

      this.saveButtonView = new ButtonView(locale);
      this.saveButtonView.set({
        label: t('Save'),
        icon: checkIcon,
        tooltip: true,
        withText: false,
        class: 'ck-button-save'
      });
      this.saveButtonView.type = 'submit';

      this.cancelButtonView = new ButtonView(locale);
      this.cancelButtonView.set({
        label: t('Cancel'),
        icon: cancelIcon,
        tooltip: true,
        withText: false,
        class: 'ck-button-cancel'
      });
      this.cancelButtonView.delegate('execute').to(this, 'cancel');

      this.setTemplate({
        tag: 'form',
        attributes: {
          class: ['ck', 'ck-media-alternative-text-form', 'ck-vertical-form'],
          tabindex: '-1'
        },
        children: [
          this.labeledInput,
          this.saveButtonView,
          this.cancelButtonView
        ]
      });
    }

    render() {
      super.render();

      submitHandler({ view: this });

      this.focusTracker.add(this.labeledInput.element);
      this.focusTracker.add(this.saveButtonView.element);
      this.focusTracker.add(this.cancelButtonView.element);

      this.keystrokes.listenTo(this.element);
    }

    destroy() {
      super.destroy();
      this.focusTracker.destroy();
      this.keystrokes.destroy();
    }

    focus() {
      this.labeledInput.focus();
    }

    getValue() {
      return this.labeledInput.fieldView.element ? this.labeledInput.fieldView.element.value : '';
    }

    setValue(value) {
      this.labeledInput.fieldView.value = value || '';
    }
  }

  class MediaTitleEditing extends Plugin {
    static get pluginName() {
      return 'MediaTitleEditing';
    }

    init() {
      var editor = this.editor;
      var schema = editor.model.schema;

      if (schema.isRegistered('drupalMedia')) {
        schema.extend('drupalMedia', {
          allowAttributes: ['drupalMediaEntityTitle']
        });
      }

      this._setupConversion();
      editor.commands.add('mediaImageTitle', new MediaImageTitleCommand(editor));
    }

    _setupConversion() {
      var editor = this.editor;
      var conversion = editor.conversion;

      conversion.for('upcast').attributeToAttribute({
        view: {
          name: 'drupal-media',
          key: 'title'
        },
        model: 'drupalMediaEntityTitle'
      });

      conversion.for('dataDowncast').attributeToAttribute({
        model: 'drupalMediaEntityTitle',
        view: 'title'
      });

      conversion.for('editingDowncast').attributeToAttribute({
        model: 'drupalMediaEntityTitle',
        view: 'title'
      });
    }
  }

  class MediaTitleUI extends Plugin {
    static get requires() {
      return [ContextualBalloon];
    }

    static get pluginName() {
      return 'MediaTitleUI';
    }

    init() {
      var editor = this.editor;
      var self = this;

      this._balloon = editor.plugins.get(ContextualBalloon);
      this._form = null;

      editor.ui.componentFactory.add('mediaImageTitle', function(locale) {
        var command = editor.commands.get('mediaImageTitle');
        var button = new ButtonView(locale);

        button.set({
          label: Drupal.t('Override media image title'),
          icon: titleIcon,
          tooltip: true
        });

        button.bind('isEnabled').to(command, 'isEnabled');

        button.on('execute', function() {
          self._showForm();
        });

        return button;
      });
    }

    destroy() {
      super.destroy();

      if (this._form) {
        this._form.destroy();
      }
    }

    _createForm() {
      var editor = this.editor;
      var form = new MediaTitleFormView(editor.locale);
      var self = this;

      this.listenTo(form, 'submit', function() {
        var value = form.getValue();
        editor.execute('mediaImageTitle', { newValue: value });
        self._hideForm();
      });

      this.listenTo(form, 'cancel', function() {
        self._hideForm();
      });

      clickOutsideHandler({
        emitter: form,
        activator: function() {
          return self._isVisible;
        },
        contextElements: function() {
          return [self._balloon.view.element];
        },
        callback: function() {
          self._hideForm();
        }
      });

      form.keystrokes.set('Esc', function(data, cancel) {
        self._hideForm();
        cancel();
      });

      return form;
    }

    _showForm() {
      if (!this._form) {
        this._form = this._createForm();
      }

      var command = this.editor.commands.get('mediaImageTitle');

      this._balloon.add({
        view: this._form,
        position: this._getBalloonPositionData()
      });

      this._form.setValue(command.value);
      this._form.focus();
    }

    _hideForm() {
      if (!this._isVisible) {
        return;
      }

      this._form.setValue('');
      this._balloon.remove(this._form);
      this.editor.editing.view.focus();
    }

    get _isVisible() {
      return this._form && this._balloon.visibleView === this._form;
    }

    _getBalloonPositionData() {
      var view = this.editor.editing.view;
      var viewDocument = view.document;

      return {
        target: function() {
          var selection = viewDocument.selection;
          var selectedElement = selection.getSelectedElement();

          if (selectedElement) {
            return view.domConverter.mapViewToDom(selectedElement);
          }

          return view.domConverter.viewRangeToDom(selection.getFirstRange());
        }
      };
    }
  }

  class MediaTitle extends Plugin {
    static get requires() {
      return [MediaTitleEditing, MediaTitleUI];
    }

    static get pluginName() {
      return 'MediaTitle';
    }
  }

  CKEditor5.mediaTitle = {
    MediaTitle: MediaTitle
  };
})(Drupal, CKEditor5);
