<?php

/**
 * @file
 * Post update file for CKEditor Standalone Styles.
 */

use Drupal\ckeditor_standalone_styles\Entity\CKEditorStyle;
use Drupal\Core\Entity\EntityStorageException;

/**
 * Convert styles to config entities.
 */
function ckeditor_standalone_styles_post_update_001_migrate_config() {
  $oldConfig = \Drupal::configFactory()->getEditable('ckeditor_standalone_styles.settings');
  $oldStyles = $oldConfig->get('styles');
  $oldStyles = trim($oldStyles);
  $oldStyles = str_replace(["\r\n", "\r"], "\n", $oldStyles);
  $parsedOldStyles = [];
  foreach (explode("\n", $oldStyles) as $oldStyle) {
    $oldStyle = trim($oldStyle);

    // Ignore empty lines in between non-empty lines.
    if (empty($oldStyle)) {
      continue;
    }

    // Validate syntax: element[.class...]|label pattern expected.
    if (!preg_match('@^ *[a-zA-Z0-9]+ *(\\.[a-zA-Z0-9_-]+ *)*\\| *.+ *$@', $oldStyle)) {
      return FALSE;
    }

    // Parse.
    [$selector, $label] = explode('|', $oldStyle);
    $classes = explode('.', $selector);
    $element = array_shift($classes);

    if (!empty($classes)) {
      $parsedOldStyles[] = [
        'label' => trim($label),
        'element' => trim($element),
        'classes' => implode("\n", $classes),
      ];
    }
  }

  $transliteration = \Drupal::transliteration();

  $logger = \Drupal::logger('ckeditor_standalone_styles');
  $defaultLangCode = \Drupal::languageManager()->getDefaultLanguage()->getId();
  $convertedCount = 0;
  $usedIds = [];
  foreach ($parsedOldStyles as $index => $parsedOldStyle) {
    // We need to generate a machine name (ID) using the label. We mimic what
    // Drupal's automatic machine name generator does here by transliterating
    // the label, making it lowercase, and swapping non alphanumeric chars with
    // underscores. We also prefix with the element type to reduce change of
    // an ID collision.
    $id = mb_strtolower($transliteration->transliterate($parsedOldStyle['label'], $defaultLangCode));
    $id = preg_replace('@[^a-z0-9_]+@', '_', $id);
    $id = strtolower($parsedOldStyle['element']) . '_' . $id;

    // Max length is 32 characters, but trim to 30 so we have room to append a
    // number for uniqueness if necessary.
    $id = substr($id, 0, 30);
    $originalId = $id;

    // It's possible this ID was already used. Make 10 attempts at creating
    // a unique one.
    $iteration = 1;
    while (in_array($id, $usedIds) && $iteration < 10) {
      $id = $originalId . '_' . $iteration;
      $iteration++;
    }

    $properties = [
      'id' => $id,
      'label' => $parsedOldStyle['label'],
      'element' => $parsedOldStyle['element'],
      'classes' => $parsedOldStyle['classes'],
      'weight' => $index,
    ];
    $styleEntity = CKEditorStyle::create($properties);
    try {
      $styleEntity->save();
      $convertedCount++;
      $usedIds[] = $id;
    }
    catch (EntityStorageException $e) {
      $logger->error('EntityStorageException when saving a CKEditor Style: %message. The style data: %data', [
        '%message' => $e->getMessage(),
        '%data' => serialize($properties),
      ]);
    }
  }

  $logger->info('Migrated %amount CKEditor Styles to config entities.', [
    '%amount' => $convertedCount,
  ]);

  $oldConfig->delete();
}
