<?php

namespace Drupal\ckeditor_standalone_styles;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Helper class for CKEditor Standalone Styles.
 */
class CKEditorStylesHelper {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * CKEditorStylesHelper constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, ModuleHandlerInterface $moduleHandler) {
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * Builds the "stylesSet" configuration part of the CKEditor4 JS settings.
   *
   * @return array
   *   An array containing the "stylesSet" configuration.
   */
  public function generateStyleSetSettingData() : array {
    $stylesSetConfig = [];

    $styleStorage = \Drupal::entityTypeManager()->getStorage('ckeditor_style');
    $query = $styleStorage
      ->getQuery()
      ->sort('weight', 'ASC');
    $styleEntities = $styleStorage->loadMultiple($query->execute());
    foreach ($styleEntities as $styleEntity) {
      /** @var \Drupal\ckeditor_standalone_styles\CKEditorStyleInterface $styleEntity */
      $classes = $styleEntity->getClasses();
      $classes = str_replace(["\r\n", "\r"], "\n", $classes);
      $classes = explode("\n", $classes);
      $classes = array_map('trim', $classes);
      $classes = array_filter($classes);

      if (!empty($classes)) {
        $stylesSetConfig[] = [
          'name' => $styleEntity->label(),
          'element' => $styleEntity->getElement(),
          'attributes' => [
            'class' => implode(' ', $classes),
          ],
        ];
      }
    }

    // Allow modules to alter the list of parsed styles.
    $this->moduleHandler->alter('ckeditor_standalone_styles', $stylesSetConfig);

    return $stylesSetConfig;
  }

  /**
   * Builds the "styles" configuration part of the CKEditor5 JS settings.
   *
   * @param array|null $allowedElements
   *   The list of HTML elements allowed by the editor's filter format.
   *   This prevents the CKE from presenting styles that apply to elements
   *   that are not allowed by the editor's filter format.
   *
   * @return array
   *   An array containing the "styles" configuration.
   */
  public function generateStyleSetSettingsDataCkeditor5(array $allowedElements = NULL): array {
    $cke4SettingsData = $this->generateStyleSetSettingData();
    $cke5SettingsData = [];
    foreach ($cke4SettingsData as $data) {
      if (isset($allowedElements) && !in_array($data['element'], $allowedElements)) {
        continue;
      }
      $cke5SettingsData[] = [
        'name' => $data['name'],
        'element' => $data['element'],
        'classes' => explode(' ', $data['attributes']['class']),
      ];
    }
    return $cke5SettingsData;
  }

}
