<?php

namespace Drupal\ckeditor_standalone_styles\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding or editing a CKEditor Style config entity.
 */
class CKEditorStyleForm extends EntityForm implements ContainerInjectionInterface {

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a CKEditorStyleForm object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(MessengerInterface $messenger) {
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\ckeditor_standalone_styles\CKEditorStyleInterface $style */
    $style = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $style->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $style->id(),
      '#machine_name' => [
        'exists' => '\Drupal\ckeditor_standalone_styles\Entity\CKEditorStyle::load',
      ],
      '#disabled' => !$style->isNew(),
    ];

    $form['element'] = [
      '#title' => $this->t('HTML Element'),
      '#type' => 'textfield',
      '#default_value' => $style->getElement(),
      '#description' => $this->t('The HTML element this style can be applied to, e.g. div, span, p.'),
      '#required' => TRUE,
    ];

    $form['classes'] = [
      '#title' => $this->t('CSS classes'),
      '#type' => 'textarea',
      '#default_value' => $style->getClasses(),
      '#description' => $this->t('One or more CSS classes that should be added to the element when this style is used. Enter one per line.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Prevent user from accidentally adding spaces around element name.
    $element = $form_state->getValue('element');
    if (!empty($element)) {
      $form_state->setValue('element', trim($element));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $style = $this->entity;
    $status = $style->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger->addStatus($this->t('Created the %label style.', [
          '%label' => $style->label(),
        ]));
        break;

      default:
        $this->messenger->addStatus($this->t('Saved the %label style.', [
          '%label' => $style->label(),
        ]));
    }
    $form_state->setRedirectUrl($style->toUrl('collection'));
  }

}
