/**
 * Method for updating ckeditor 4 data to ckeditor 5 data.
 *
 * @param {String} tooltip
 *   A string containing tooltip information.
 *
 * @return {String|null}
 *   Ckeditor 5 compatible string.
 */
function handleDataMigration(tooltip) {
  // Check that there is values being parsed
  if (!tooltip) {
    return null;
  }

  return tooltip;
}

/**
 * Method for creating an arbitery tooltip.
 *
 * @param {string} tooltipData
 *   The tooltip data definition.
 *
 * @param {Object} writer
 *   The ckeditor writer
 *
 * @return {Element}
 *   The tooltip element.
 */
function createTooltipElement(tooltip, {writer}) {
  const tooltipElement = writer.createAttributeElement(
    'tippy',
    {
      'data-tippy-content': tooltip,
      class: 'tippy-tooltip-text',
    },
    {priority: 6},
  );

  writer.setCustomProperty('tooltip', true, tooltipElement);
  return tooltipElement;
}

/**
 * Check if a node is a tooltip.
 *
 * @param {ViewNode|ViewDocumentFragment} node
 *   The node to be chcked
 * @return {Boolean}
 *   Returns `true` if a given view node is the link element.
 */
function isTooltipElement(node) {
  return node.is('attributeElement') && !!node.getCustomProperty('tooltip');
}

/**
 *
 * @param {HTMLElement} element
 *   The element to check.
 *
 * @param {Object} schema
 *   The tooltip schema.
 *
 * @return {Boolean}
 *   Whether this is a tooltip element.
 */
function isTooltipableElement(element, schema) {
  if (!element) {
    return false;
  }

  return schema.checkAttribute(element.name, 'data-tippy-content');
}

/**
 * A helper function that retrieves and concatenates all text within the model range.
 */
function getRangeText(range) {
  return Array.from(range.getItems()).reduce((rangeText, node) => {
    if (!(node.is('text') || node.is('textProxy'))) {
      return rangeText;
    }

    return rangeText + node.data;
  }, '');
}

function extractTooltipFromSelection(selection) {
  if (selection.isCollapsed) {
    const firstPosition = selection.getFirstPosition();

    if (!firstPosition) {
      return null;
    }

    return firstPosition.textNode && firstPosition.textNode.data;
  }

  const firstRange = selection.getFirstRange();

  if (!firstRange) {
    return null;
  }

  const rangeItems = Array.from(selection.getFirstRange().getItems());

  if (rangeItems.length > 1) {
    return null;
  }

  const firstNode = rangeItems[0];

  if (firstNode.is('$text') || firstNode.is('$textProxy')) {
    return firstNode.data;
  }

  return null;
}

export {
  createTooltipElement,
  isTooltipableElement,
  isTooltipElement,
  handleDataMigration,
  getRangeText,
  extractTooltipFromSelection,
};
