/* eslint-disable class-methods-use-this */
/* eslint-disable no-restricted-syntax */
import {Command} from 'ckeditor5/src/core';
import {first, toMap} from 'ckeditor5/src/utils';
import {findAttributeRange} from 'ckeditor5/src/typing';
import {isTooltipableElement, getRangeText, extractTooltipFromSelection} from '../utils';

export default class AddTooltipCommand extends Command {
  /**
   * The value of the `'data-tooltip'` attribute if the start of the selection is located in a node with this attribute.
   *
   * @observable
   * @readonly
   */
  value = undefined;

  refresh() {
    const {model} = this.editor;
    const {selection} = model.document;
    const selectedElement = selection.getSelectedElement() || first(selection.getSelectedBlocks());
    const firstRange = selection.getFirstRange();
    const position = selection.getFirstPosition();

    let tooltipRange = findAttributeRange(
      position,
      'data-tippy-content',
      selection.getAttribute('data-tippy-content'),
      model,
    );

    if (isTooltipableElement(selectedElement, model.schema)) {
      this.value = selectedElement.getAttribute('data-tippy-content');
      this.isEnabled = model.schema.checkAttribute(selectedElement, 'data-tippy-content');
    } else {
      this.value = selection.getAttribute('data-tippy-content');
      this.value = {
        text: getRangeText(tooltipRange),
        data: selection.getAttribute('data-tippy-content'),
      };
      this.isEnabled = model.schema.checkAttributeInSelection(selection, 'data-tippy-content');
    }
  }

  execute(tooltip) {
    // Create object that contains supported data-attributes in view data by
    // flipping `DrupalMediaEditing.attrs` object (i.e. keys from object become
    // values and values from object become keys).
    const dataAttributeMapping = Object.entries({
      tooltipText: 'data-text',
      tooltipTitle: 'data-title',
    }).reduce(
      (result, [key, value]) => {
        result[value] = key;
        return result;
      },
      {},
    );

    // \Drupal\media\Form\EditorMediaDialog returns data in keyed by
    // data-attributes used in view data. This converts data-attribute keys to
    // keys used in model.
    const modelAttributes = Object.keys(tooltip).reduce(
      (result, attribute) => {
        if (dataAttributeMapping[attribute]) {
          if (typeof tooltip[attribute].value !== 'undefined') {
            result[dataAttributeMapping[attribute]] =
              tooltip[attribute].value;
          } else {
            result[dataAttributeMapping[attribute]] = tooltip[attribute];
          }
        }
        return result;
      },
      {},
    );

    const {model} = this.editor;
    const {selection} = model.document;

    // Set the tooltip from body-text.
    tooltip = modelAttributes.tooltipText;

    model.change((writer) => {
      if (selection.isCollapsed) {
        const position = selection.getFirstPosition();

        if (selection.hasAttribute('data-tippy-content')) {
          const extractedTooltipText = extractTooltipFromSelection(selection);
          let tooltipRange = findAttributeRange(
            position,
            'data-tippy-content',
            selection.getAttribute('data-tippy-content'),
            model,
          );

          if (selection.getAttribute('data-tippy-content') === extractedTooltipText) {
            tooltipRange = this._updateTooltipContent(model, writer, tooltipRange, tooltip);
          }

          writer.setAttribute('data-tippy-content', tooltip, tooltipRange);

          // Put the selection at the end of the updated link.
          writer.setSelection(writer.createPositionAfter(tooltipRange.end.nodeBefore));
        } else if (tooltip !== '') {
          const attributes = toMap(selection.getAttributes());
          attributes.set('data-tippy-content', tooltip);
          const {end: positionAfter} = model.insertContent(
            writer.createText(modelAttributes.tooltipTitle, attributes),
            position,
          );

          // Put the selection at the end of the inserted link.
          // Using end of range returned from insertContent in case nodes with the same attributes got merged.
          writer.setSelection(positionAfter);
        }

        ['data-tippy-content'].forEach((item) => {
          writer.removeSelectionAttribute(item);
        });
      } else {
        const ranges = model.schema.getValidRanges(selection.getRanges(), 'data-tippy-content');
        const allowedRanges = [];

        for (const element of selection.getSelectedBlocks()) {
          if (model.schema.checkAttribute(element, 'data-tippy-content')) {
            allowedRanges.push(writer.createRangeOn(element));
          }
        }

        const rangesToUpdate = allowedRanges.slice();
        for (const range of ranges) {
          if (this._isRangeToUpdate(range, allowedRanges)) {
            rangesToUpdate.push(range);
          }
        }

        // Remove tooltip if the text is empty.
        if (modelAttributes.tooltipText === '' && modelAttributes.tooltipTitle !== '') {
          for (const range of rangesToUpdate) {
            writer.removeAttribute('data-tippy-content', range);
          }
        } else {
          for (const range of rangesToUpdate) {
            let tooltipRange = range;
            if (rangesToUpdate.length === 1) {
              // Current text of the link in the document.
              const extractedTooltipText = extractTooltipFromSelection(selection);
              if (selection.getAttribute('data-tippy-content') === extractedTooltipText) {
                tooltipRange = this._updateTooltipContent(model, writer, range, tooltip);
                writer.setSelection(writer.createSelection(tooltipRange));
              }
            }
            writer.setAttribute('data-tippy-content', tooltip, tooltipRange);
          }
        }


      }
    });
  }

  /**
   * Checks whether specified `range` is inside an element that accepts the `linkHref` attribute.
   *
   * @param {Range} range
   *   A range to check.
   *
   * @param {Range[]} allowedRanges
   *   An array of ranges created on elements where the attribute is accepted.
   *
   * @return {Boolean}
   *  Wehter range is in allowedRanges
   */
  _isRangeToUpdate(range, allowedRanges) {
    for (const allowedRange of allowedRanges) {
      // A range is inside an element that will have the `linkHref` attribute. Do not modify its nodes.
      if (allowedRange.containsRange(range)) {
        return false;
      }
    }

    return true;
  }

  /**
   * Updates selected tootlip with a new value as its content and as its data-tooltip attribute.
   *
   * @param {Model} model
   *   Model is needed to insert content.
   *
   * @param {Writer} writer
   *   Writer is needed to create text element in model.
   *
   * @param {Range} range
   *   A range where the new content should be instered,
   *
   * @param {String} tooltip
   *   The tooltip vlaue which should be in the data-tooltip attribute in the content.
   *
   * @return {Range}
   *   The updated range
   */
  _updateTooltipContent(model, writer, range, tooltip) {
    const text = writer.createText(tooltip, {'data-tippy-content': tooltip});
    return model.insertContent(text, range);
  }
}
