import {Plugin} from 'ckeditor5/src/core';
import {ButtonView} from 'ckeditor5/src/ui';
import {isWidget} from 'ckeditor5/src/widget';
import {ClickObserver} from 'ckeditor5/src/engine';
import {isTooltipableElement, extractTooltipFromSelection} from './utils';
import icon from '../../../../icons/comment-alt-regular.svg';
import {DomEventObserver} from "ckeditor5/src/engine";

/**
 * Ckeditor5 doesn't support double click out of the box.
 * Register it here so we can use it.
 *
 * @Todo Replace double click with a balloon style popup menu to
 *   edit the embedded content item.
 */
class DoubleClickObserver extends DomEventObserver {
  constructor(view) {
    super(view);
    this.domEventType = 'dblclick';
  }

  onDomEvent(domEvent) {
    this.fire(domEvent.type, domEvent);
  }
}

/**
 * Returns a tooltip element if there's one among the ancestors of the provided `Position`.
 *
 * @param {ViewPosition} position
 *   View position to analyze.
 * @return {ViewAttributeElement|null}
 *   Link element at the position or null.
 */
function findTooltipElementAncestor(position) {
  return position.getAncestors().find((ancestor) => isTooltipElement(ancestor)) || null;
}

export default class Tippytooltip_ui extends Plugin {

  /**
   * @inheritDoc
   */
  static get pluginName() {
    return 'TooltipUI';
  }

  /**
   * @inheritDoc
   */
  init() {
    const editor = this.editor;
    const options = this.editor.config.get('tippyTooltip');
    if (!options) {
      return;
    }

    const {dialogURL, openDialog, dialogSettings = {}} = options;
    if (!dialogURL || typeof openDialog !== 'function') {
      return;
    }
    editor.ui.componentFactory.add('TippyTooltip', (locale) => {
      const command = editor.commands.get('add-tooltip');
      const buttonView = new ButtonView(locale);
      buttonView.set({
        label: Drupal.t('Tooltip'),
        icon: icon,
        tooltip: true,
      });

      // Bind the state of the button to the command.
      buttonView.bind('isOn', 'isEnabled').to(command, 'value', 'isEnabled');

      this.listenTo(buttonView, 'execute', () => {
        const {view} = this.editor.editing;
        const {selection} = view.document;
        const selectedElement = selection.getSelectedElement();
        const url = new URL(dialogURL, document.baseURI);
        const range = editor.model.document.selection.getFirstRange();
        const rangeItems = Array.from(editor.model.document.selection.getFirstRange().getItems());
        const firstNode = rangeItems[0];

        if (selectedElement && typeof selectedElement.name !== 'undefined' && selectedElement.name === 'tippy') {
          url.searchParams.append(
            'content',
            selectedElement.getAttribute('data-tippy-content'),
          );
          url.searchParams.append(
            'text',
            //extractTooltipFromSelection(selection),
            selectedElement._children[0]._textData,
          );
        } else {
          url.searchParams.append(
            'content',
            ''
          );
          url.searchParams.append(
            'text',
            (firstNode === undefined) ? '' : firstNode.data,
          );
        }
        openDialog(
          url.toString(),
          ({attributes}) => {
            editor.execute('add-tooltip', attributes);
          },
          dialogSettings,
        );
      });

      return buttonView;
    });

    const view = editor.editing.view;
    const viewDocument = view.document;
    view.addObserver(DoubleClickObserver);

    editor.listenTo(viewDocument, 'dblclick', (evt, data) => {
      const modelElement = editor.editing.mapper.toModelElement(
        data.domTarget,
      );

      if (
        data.domTarget &&
        typeof data.domTarget.nodeName !== 'undefined' &&
        data.domTarget.localName === 'tippy'
      ) {
        const query = {
          content: data.domTarget.getAttribute('data-tippy-content'),
          text: data.domTarget.innerText,
        };
        openDialog(
          `${dialogURL}?${new URLSearchParams(query)}`,
          ({attributes}) => {
            editor.execute('add-tooltip', attributes);
          },
          dialogSettings,
        );
      }
    });
  }

  /**
   * Returns the link {@link module:engine/view/attributeelement~AttributeElement} under
   * the {@link module:engine/view/document~Document editing view's} selection or `null`
   * if there is none.
   *
   * **Note**: For a nonÃ¢â‚¬â€œcollapsed selection, the link element is returned when **fully**
   * selected and the **only** element within the selection boundaries, or when
   * a linked widget is selected.
   *
   * @return {ViewAttributeElement|null}
   *   The selected tooltip element or null.
   */
  _getSelectedTooltipElement() {
    const {view} = this.editor.editing;
    const {selection} = view.document;
    const selectedElement = selection.getSelectedElement();

    if (selection.isCollapsed || (selectedElement && isWidget(selectedElement))) {
      return findTooltipElementAncestor(selection.getFirstPosition());
    }

    if (selection.getFirstRange()) {
      const range = selection.getFirstRange().getTrimmed();
      const startTooltip = findTooltipElementAncestor(range.start);
      const endTooltip = findTooltipElementAncestor(range.end);

      if (!startTooltip || startTooltip !== endTooltip) {
        return null;
      }

      if (view.createRangeIn(startTooltip).getTrimmed().isEqual(range)) {
        return startTooltip;
      }
    }

    return null;
  }
}
