<?php

namespace Drupal\ckeditor_tippy\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\editor\Ajax\EditorDialogSave;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Provides a link dialog for text editors.
 *
 * @internal
 */
class TooltipEditorDialog extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'editor_tooltip_dialog';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // The default values are set directly from \Drupal::request()->request,
    // provided by the editor plugin opening the dialog.
    $user_input = $form_state->getUserInput();
    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'editor/drupal.editor.dialog';
    $form['#prefix'] = '<div id="editor-tooltip-dialog-form">';
    $form['#suffix'] = '</div>';
    $has_text = isset($user_input['editor_object']['text']) && $user_input['editor_object']['text'];
    $title = $has_text ? $this->t('Selected text') : $this->t('Insert new text');
    $form['attributes']['body_text'] = [
      '#title' => $title,
      '#type' => 'textfield',
      '#default_value' => $this->getRequest()->query->get('text') ?? '',
      '#disabled' => $has_text ? TRUE : FALSE,
      '#maxlength' => 2048,
      '#description' => $has_text ? $this->t('Tooltip is added to this text.') : $this->t('Insert new text to be added at the current cursor position.'),
    ];

    // Everything under the "attributes" key is merged directly into the
    // generated link tag's attributes.
    $form['attributes']['tooltip_text'] = [
      '#title' => $this->t('Tooltip text'),
      '#type' => 'textfield',
      '#default_value' => $this->getRequest()->query->get('content') ?? '',
      '#maxlength' => 2048,
      '#description' => $this->t('Enter the text which appears on mouse hover. Leave empty to remove the tooltip.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['save_modal'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      // No regular submit-handler. This form only works via JavaScript.
      '#submit' => [],
      '#ajax' => [
        'callback' => '::submitForm',
        'event' => 'click',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    if ($form_state->getErrors()) {
      unset($form['#prefix'], $form['#suffix']);
      $form['status_messages'] = [
        '#type' => 'status_messages',
        '#weight' => -10,
      ];
      $response->addCommand(new HtmlCommand('#editor-tooltip-dialog-form', $form));
    }
    else {
      if (isset($form_state->getValue(['attributes', 'tooltip_text'])['value'])) {
        $markup = check_markup($form_state->getValue(['attributes', 'tooltip_text'])['value'], $form_state->getValue(['attributes', 'tooltip_text'])['format']);
        if (!empty($form_state->getValue(['attributes', 'tooltip_text'])['value'])) {
          $form_state->setValue(['attributes', 'tooltip_text'], $markup->__toString());
        }
      }

      $response->addCommand(new EditorDialogSave([
        'attributes' => [
          'data-title' => $form_state->getValue(['attributes', 'body_text']),
          'data-text' => $form_state->getValue(['attributes', 'tooltip_text']) ?? '',
        ],
      ]));

      $response->addCommand(new CloseModalDialogCommand());
    }

    return $response;
  }

  /**
   * Access callback for viewing the dialog.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   *
   * @return \Drupal\Core\Access\AccessResult|\Drupal\Core\Access\AccessResultReasonInterface
   *   The acccess result.
   */
  public function checkAccess(AccountProxyInterface $account, $editor) {
    return AccessResult::allowedIfHasPermission($account, 'use text format ' . $editor);
  }

}