(function () {
  'use strict';

  function drupalBaseUrl() {
    if (window.drupalSettings && drupalSettings.path) {
      return drupalSettings.path.baseUrl || drupalSettings.path.basePath || '/';
    }
    return '/';
  }

  function guessMime(url) {
    var u = (url || '').toLowerCase();
    if (u.indexOf('.webm') > -1) return 'video/webm';
    if (u.indexOf('.ogg') > -1 || u.indexOf('.ogv') > -1) return 'video/ogg';
    return 'video/mp4';
  }

  function getSavedAttr(el, attr) {
    if (!el) return '';
    return el.getAttribute('data-cke-saved-' + attr) || el.getAttribute(attr) || '';
  }

  function setSavedAttr(el, attr, value) {
    if (!el) return;
    if (value) {
      el.setAttribute(attr, value);
      el.setAttribute('data-cke-saved-' + attr, value);
    } else {
      el.removeAttribute(attr);
      el.removeAttribute('data-cke-saved-' + attr);
    }
  }

  function openImceFillInput(dialog, tabId, fieldId, type) {
    var inputEl = dialog.getContentElement(tabId, fieldId).getInputElement();
    if (!inputEl || !inputEl.$ || !inputEl.$.id) return;

    var domId = inputEl.$.id;
    if (window.imceInput && typeof window.imceInput.openImce === 'function') {
      window.imceInput.openImce('imceInput.urlSendto', type || 'link', 'inputId=' + domId);
      return;
    }

    var url = drupalBaseUrl() + 'imce?urlField=' + encodeURIComponent(domId);
    if (type) {
      url += '&type=' + encodeURIComponent(type);
    }
    window.open(url, '', 'width=760,height=560,resizable=1');
  }

  // Robustly locate the <video> element the user clicked/placed caret near.
  function findActiveVideo(editor) {
    var sel = editor.getSelection();
    if (!sel) return null;

    // 1) Selected element (sometimes null for object elements)
    var el = sel.getSelectedElement();
    if (el) {
      if (el.is('video')) return el;
      if (el.is('source')) {
        var vFromSource = el.getAscendant('video', true);
        if (vFromSource) return vFromSource;
      }
      var vFromSelected = el.getAscendant && el.getAscendant('video', true);
      if (vFromSelected) return vFromSelected;

      // If CKEditor created a fake element, try restoring the real element.
      if (el.getAttribute && el.getAttribute('data-cke-realelement')) {
        try {
          var real = editor.restoreRealElement(el);
          if (real && real.is && real.is('video')) return real;
        } catch (e) {}
      }
    }

    // 2) Start element (common when caret is inside/near)
    var start = sel.getStartElement && sel.getStartElement();
    if (start) {
      if (start.is('video')) return start;
      var vFromStart = start.getAscendant && start.getAscendant('video', true);
      if (vFromStart) return vFromStart;

      // 3) elementPath is often best for object-ish elements
      try {
        var path = editor.elementPath(start);
        if (path && path.contains) {
          var vFromPath = path.contains('video', true);
          if (vFromPath) return vFromPath;
        }
      } catch (e2) {}
    }

    // 4) Range ancestor fallback (when click lands next to the video)
    try {
      var ranges = sel.getRanges();
      if (ranges && ranges.length) {
        var anc = ranges[0].getCommonAncestor(true);
        if (anc) {
          if (anc.is && anc.is('video')) return anc;
          var vFromAnc = anc.getAscendant && anc.getAscendant('video', true);
          if (vFromAnc) return vFromAnc;
        }
      }
    } catch (e3) {}

    return null;
  }

  function readVideoData(videoEl) {
    var data = {
      video: '',
      poster: '',
      autoplay: false,
      controls: false,
      loop: false,
      height: '',
      width: ''
    };

    if (!videoEl) return data;

    // poster can be stored as poster or data-cke-saved-poster (depending on filters)
    data.poster = getSavedAttr(videoEl, 'poster');
    data.autoplay = !!getSavedAttr(videoEl, 'autoplay');
    data.controls = !!getSavedAttr(videoEl, 'controls');
    data.loop = !!getSavedAttr(videoEl, 'loop');
    data.height = getSavedAttr(videoEl, 'height');
    data.width = getSavedAttr(videoEl, 'width');

    // Prefer first <source>
    var source = videoEl.findOne && videoEl.findOne('source');
    if (source) {
      data.video = getSavedAttr(source, 'src');
    } else {
      // Rare: <video src="...">
      data.video = getSavedAttr(videoEl, 'src');
    }

    return data;
  }

  function ensureSource(videoEl, editor) {
    var source = videoEl.findOne && videoEl.findOne('source');
    if (source) return source;

    source = editor.document.createElement('source');
    videoEl.append(source);
    return source;
  }

  CKEDITOR.dialog.add('videoimceDialog', function (editor) {
    function validateSize() {
      return function () {
        var value = this.getValue();
        if (value === '') {
          return true;
        }
        return /^(\d+|\d+%|\d+px)$/.test(value) ? true : 'Use numbers, or a number with % or px.';
      };
    }

    return {
      title: 'Video',
      minWidth: 520,
      minHeight: 160,

      contents: [
        {
          id: 'info',
          label: 'Settings',
          elements: [
            {
              type: 'hbox',
              widths: ['80%', '20%'],
              children: [
                {
                  type: 'text',
                  id: 'video',
                  label: 'Video',
                  required: true,
                  validate: CKEDITOR.dialog.validate.notEmpty('Video URL is required.')
                },
                {
                  type: 'button',
                  id: 'browseVideo',
                  label: editor.lang.common.browseServer || 'Browse',
                  onClick: function () {
                    openImceFillInput(this.getDialog(), 'info', 'video', 'link');
                  }
                }
              ]
            },
            {
              type: 'hbox',
              widths: ['33%', '33%', '34%'],
              children: [
                {
                  type: 'checkbox',
                  id: 'autoplay',
                  label: 'Autoplay'
                },
                {
                  type: 'checkbox',
                  id: 'controls',
                  label: 'Controls'
                },
                {
                  type: 'checkbox',
                  id: 'loop',
                  label: 'Loop'
                }
              ]
            },
            {
              type: 'hbox',
              widths: ['50%', '50%'],
              children: [
                {
                  type: 'text',
                  id: 'width',
                  label: 'Width',
                  validate: validateSize()
                },
                {
                  type: 'text',
                  id: 'height',
                  label: 'Height',
                  validate: validateSize()
                }
              ]
            },
            {
              type: 'hbox',
              widths: ['80%', '20%'],
              children: [
                {
                  type: 'text',
                  id: 'poster',
                  label: 'Poster'
                },
                {
                  type: 'button',
                  id: 'browsePoster',
                  label: editor.lang.common.browseServer || 'Browse',
                  onClick: function () {
                    openImceFillInput(this.getDialog(), 'info', 'poster', 'image');
                  }
                }
              ]
            }
          ]
        }
      ],

      onShow: function () {
        // Find the real <video> the user clicked (or is next to).
        var videoEl = editor._videoimceSelectedElement || findActiveVideo(editor);

        this._videoEl = videoEl || null;

        var data = readVideoData(this._videoEl);

        // Populate fields
        this.setValueOf('info', 'video', data.video || '');
        this.setValueOf('info', 'poster', data.poster || '');
        this.setValueOf('info', 'autoplay', data.autoplay);
        this.setValueOf('info', 'controls', data.controls);
        this.setValueOf('info', 'loop', data.loop);
        this.setValueOf('info', 'width', data.width || '');
        this.setValueOf('info', 'height', data.height || '');

        // Clear cached element after use to avoid stale references.
        editor._videoimceSelectedElement = null;
      },

      onOk: function () {
        var videoUrl = this.getValueOf('info', 'video');
        var posterUrl = this.getValueOf('info', 'poster');
        var autoplay = this.getValueOf('info', 'autoplay');
        var controls = this.getValueOf('info', 'controls');
        var loop = this.getValueOf('info', 'loop');
        var width = this.getValueOf('info', 'width');
        var height = this.getValueOf('info', 'height');

        var videoEl = this._videoEl;

        // Create if not editing an existing <video>
        if (!videoEl) {
          videoEl = editor.document.createElement('video');

          // Reasonable defaults (keep if you want)
          videoEl.setAttribute('controls', 'controls');
          videoEl.setAttribute('crossorigin', 'anonymous');
          videoEl.setAttribute('preload', 'auto');
          videoEl.setAttribute('style', 'max-width:100%; height:auto;');
        }

        // Poster: set both poster + data-cke-saved-poster
        setSavedAttr(videoEl, 'poster', posterUrl);
        setSavedAttr(videoEl, 'autoplay', autoplay ? 'autoplay' : '');
        setSavedAttr(videoEl, 'controls', controls ? 'controls' : '');
        setSavedAttr(videoEl, 'loop', loop ? 'loop' : '');
        setSavedAttr(videoEl, 'width', width);
        setSavedAttr(videoEl, 'height', height);

        // Ensure <source> and update src/type (set both src + data-cke-saved-src)
        var sourceEl = ensureSource(videoEl, editor);
        setSavedAttr(sourceEl, 'src', videoUrl);

        var mime = guessMime(videoUrl);
        if (mime) sourceEl.setAttribute('type', mime);

        // Insert new element if needed
        if (!this._videoEl) {
          editor.insertElement(videoEl);
        }
      }
    };
  });

})();
