/**
 * @file
 * CL Preview JavaScript.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.sdcPreview = {
    attach: function (context, settings) {
      // Auto-resize ALL iframes to fit content (not just the first one)
      const iframes = context.querySelectorAll('.cl-preview-iframe');

      iframes.forEach(function(iframe) {
        if (iframe.dataset.resizeInitialized) {
          return;
        }

        iframe.dataset.resizeInitialized = 'true';

        // Function to resize iframe to content
        const setupResizing = function() {
          try {
            const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;

            if (!iframeDoc || !iframeDoc.body) {
              return;
            }

            const resizeIframe = function() {
              // Get the actual content height from the main content block
              const mainBlock = iframeDoc.querySelector('.block-system-main-block');

              let height;
              if (mainBlock) {
                // Use the main block's height
                height = Math.max(
                  mainBlock.scrollHeight,
                  mainBlock.offsetHeight,
                  mainBlock.clientHeight
                );
              } else {
                // Fallback to body/html if main block not found
                const body = iframeDoc.body;
                const html = iframeDoc.documentElement;

                height = Math.max(
                  body.scrollHeight,
                  body.offsetHeight,
                  html.clientHeight,
                  html.scrollHeight,
                  html.offsetHeight
                );
              }

              iframe.style.height = height + 'px';
              console.log('Resized iframe to:', height + 'px', iframe.dataset.componentId, mainBlock ? '(from .block-system-main-block)' : '(from body/html)');
            };

            // Initial resize
            resizeIframe();

            // Retry after a short delay to ensure all content is loaded
            setTimeout(resizeIframe, 100);
            setTimeout(resizeIframe, 300);
            setTimeout(resizeIframe, 500);

            // Watch for content changes
            const observer = new MutationObserver(resizeIframe);
            observer.observe(iframeDoc.body, {
              childList: true,
              subtree: true,
              attributes: true,
              characterData: true
            });

            // Also watch for window resize
            if (iframeDoc.defaultView) {
              iframeDoc.defaultView.addEventListener('resize', resizeIframe);
            }
          } catch (e) {
            // Cross-origin iframe, can't access content
            console.log('Cannot resize iframe:', e.message);
          }
        };

        // Setup resizing on load
        iframe.addEventListener('load', setupResizing);

        // If iframe is already loaded (has src), setup resizing immediately
        if (iframe.src && iframe.contentDocument) {
          setupResizing();
        }
      });
    }
  };


  /**
   * Lazy loading behavior for component iframes.
   */
  Drupal.behaviors.clPreviewLazyLoad = {
    attach: function (context, settings) {
      // Find all iframes that should be lazy loaded
      const iframes = context.querySelectorAll('iframe.cl-preview-iframe[data-lazy-src]');

      if (iframes.length === 0 || !('IntersectionObserver' in window)) {
        return;
      }

      // Create intersection observer
      const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            const iframe = entry.target;
            const lazySrc = iframe.getAttribute('data-lazy-src');

            if (lazySrc && !iframe.src) {
              console.log('Lazy loading iframe for:', iframe.dataset.componentId);
              iframe.src = lazySrc;
              observer.unobserve(iframe);
            }
          }
        });
      }, {
        root: null,
        rootMargin: '50px', // Start loading 50px before entering viewport
        threshold: 0.01
      });

      // Observe each iframe
      iframes.forEach(iframe => {
        if (!iframe.dataset.lazyObserved) {
          iframe.dataset.lazyObserved = 'true';
          observer.observe(iframe);
        }
      });
    }
  };

  /**
   * Expand/Collapse all components behavior.
   */
  Drupal.behaviors.clPreviewExpandCollapse = {
    attach: function (context, settings) {
      const expandAllBtn = context.querySelector('.cl-preview-expand-all');
      const collapseAllBtn = context.querySelector('.cl-preview-collapse-all');

      if (!expandAllBtn || !collapseAllBtn) {
        return;
      }

      if (expandAllBtn.dataset.initialized || collapseAllBtn.dataset.initialized) {
        return;
      }

      expandAllBtn.dataset.initialized = 'true';
      collapseAllBtn.dataset.initialized = 'true';

      expandAllBtn.addEventListener('click', function(e) {
        e.preventDefault();
        const details = document.querySelectorAll('.cl-preview-component-wrapper details');
        details.forEach(detail => {
          detail.open = true;
        });
      });

      collapseAllBtn.addEventListener('click', function(e) {
        e.preventDefault();
        const details = document.querySelectorAll('.cl-preview-component-wrapper details');
        details.forEach(detail => {
          detail.open = false;
        });
      });
    }
  };

})(Drupal, drupalSettings);
