/**
 * @file
 * Hot reload functionality for CL Preview.
 *
 * Polls the server to check if component files have changed and automatically
 * reloads the iframe preview when changes are detected.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.sdcPreviewHotReload = {
    attach: function (context) {
      const iframe = context.querySelector('.cl-preview-iframe');
      if (!iframe || iframe.dataset.hotReloadInitialized) {
        return;
      }

      iframe.dataset.hotReloadInitialized = 'true';

      // Get component ID from iframe data attribute
      const componentId = iframe.dataset.componentId;
      if (!componentId) {
        return;
      }

      let lastModified = null;
      let intervalId = null;

      // Poll interval in milliseconds (1 second)
      const pollInterval = 1000;

      // Check for file changes
      const checkForChanges = async () => {
        try {
          const response = await fetch(`/cl-preview/check-updates/${componentId}`, {
            method: 'GET',
            headers: {
              'X-Requested-With': 'XMLHttpRequest'
            }
          });

          if (!response.ok) return;

          const data = await response.json();

          if (lastModified === null) {
            // First check, just store the timestamp
            lastModified = data.last_modified;
          } else if (data.last_modified > lastModified) {
            // Files changed, reload iframe
            console.log('Component files changed, reloading preview...');
            iframe.contentWindow.location.reload();
            lastModified = data.last_modified;

            // Show notification
            showNotification('Preview reloaded - files changed');
          }
        } catch (error) {
          console.error('Hot reload check failed:', error);
        }
      };

      // Show notification
      const showNotification = (message) => {
        const notification = document.createElement('div');
        notification.className = 'cl-preview-notification';
        notification.textContent = message;
        document.body.appendChild(notification);

        setTimeout(() => {
          notification.classList.add('cl-preview-notification--visible');
        }, 10);

        setTimeout(() => {
          notification.classList.remove('cl-preview-notification--visible');
          setTimeout(() => notification.remove(), 300);
        }, 2000);
      };

      // Get checkbox
      const hotReloadCheckbox = context.querySelector('input[name="hot_reload"]');
      if (!hotReloadCheckbox) {
        return;
      }

      // Toggle function
      const toggleHotReload = (enabled) => {
        if (enabled) {
          // Start polling
          if (!intervalId) {
            intervalId = setInterval(checkForChanges, pollInterval);
            checkForChanges(); // Check immediately when enabled
          }
          showNotification('Hot reload enabled');
        } else {
          // Stop polling
          if (intervalId) {
            clearInterval(intervalId);
            intervalId = null;
          }
          showNotification('Hot reload disabled');
        }
      };

      // Listen to checkbox change
      hotReloadCheckbox.addEventListener('change', (e) => {
        toggleHotReload(e.target.checked);
      });

      // Initialize based on default value
      if (hotReloadCheckbox.checked) {
        toggleHotReload(true);
      }

      // Cleanup on page unload
      window.addEventListener('beforeunload', () => {
        if (intervalId) {
          clearInterval(intervalId);
        }
      });
    }
  };

  /**
   * Handle preview variation switching for components with multiple previews.
   */
  Drupal.behaviors.sdcPreviewVariationSelector = {
    attach: function (context) {
      const variationSelector = context.querySelector('.preview-variation-selector');
      const iframe = context.querySelector('.cl-preview-iframe');

      if (!variationSelector || !iframe || variationSelector.dataset.initialized) {
        return;
      }

      variationSelector.dataset.initialized = 'true';

      // Listen to variation selector change
      variationSelector.addEventListener('change', (e) => {
        const selectedVariation = e.target.value;
        const currentSrc = iframe.src;
        const url = new URL(currentSrc);

        // Update the variation query parameter
        url.searchParams.set('variation', selectedVariation);

        // Reload iframe with new variation
        iframe.src = url.toString();
      });
    }
  };

})(Drupal, drupalSettings);
