<?php

namespace Drupal\cl_preview\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\cl_preview\Service\ComponentDiscoveryService;
use Drupal\cl_preview\Service\ComponentHelperService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the all-in-one components page.
 */
class AllComponentsController extends ControllerBase {

  /**
   * The component discovery service.
   *
   * @var \Drupal\cl_preview\Service\ComponentDiscoveryService
   */
  protected $componentDiscovery;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The component helper service.
   *
   * @var \Drupal\cl_preview\Service\ComponentHelperService
   */
  protected $componentHelper;

  /**
   * Constructs an AllComponentsController object.
   */
  public function __construct(
    ComponentDiscoveryService $component_discovery,
    ConfigFactoryInterface $config_factory,
    ComponentHelperService $component_helper
  ) {
    $this->componentDiscovery = $component_discovery;
    $this->configFactory = $config_factory;
    $this->componentHelper = $component_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cl_preview.component_discovery'),
      $container->get('config.factory'),
      $container->get('cl_preview.component_helper')
    );
  }

  /**
   * Builds the all components page.
   */
  public function build() {
    $config = $this->config('cl_preview.settings');
    $scan_themes = $config->get('scan_themes') ?: [];
    $scan_modules = $config->get('scan_modules') ?: [];

    // Check if no themes or modules are configured.
    if (empty($scan_themes) && empty($scan_modules)) {
      $settings_url = Url::fromRoute('cl_preview.settings');

      $build['empty_message'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No themes or modules are configured for scanning. Please <a href="@settings_url">configure the module settings</a> to select which themes and modules should be scanned for components.', [
            '@settings_url' => $settings_url->toString(),
          ]),
        ],
      ];

      return $build;
    }

    $grouped_components = $this->componentDiscovery->getGroupedComponents();

    // Check if no components were found even though themes/modules are configured.
    if (empty($grouped_components)) {
      $build['no_components'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No components found in the configured themes and modules. Make sure the selected themes/modules contain a <code>components/</code> directory with <code>*.component.yml</code> files.'),
        ],
      ];

      return $build;
    }

    $build['#attached']['library'][] = 'cl_preview/library';

    // Get pinned components.
    $pinned_components = $config->get('pinned_components') ?: [];

    // Filter only pinned components.
    $pinned_grouped_components = [];
    foreach ($grouped_components as $group => $components) {
      foreach ($components as $plugin_id => $definition) {
        // Only include if pinned.
        if (in_array($plugin_id, $pinned_components)) {
          $pinned_grouped_components[$group][$plugin_id] = $definition;
        }
      }
    }

    // Group components by provider.
    $by_provider = $this->componentHelper->groupComponentsByProvider($pinned_grouped_components);

    // Build component previews organized by provider and group.
    foreach ($by_provider as $provider => $groups) {
      $provider_key = 'provider_' . $this->componentHelper->generateKey($provider);
      $provider_label = $this->componentHelper->getProviderLabel($provider);

      foreach ($groups as $group => $components) {
        $group_key = $provider_key . '_' . $this->componentHelper->generateKey($group);

        $build[$group_key] = [
          '#type' => 'details',
          '#title' => $provider_label . ' (' . $group . ')',
          '#open' => TRUE,
        ];

        foreach ($components as $plugin_id => $definition) {
          $component_key = $this->componentHelper->generateKey($plugin_id);

          $build[$group_key][$component_key] = $this->buildComponentPreview($plugin_id, $definition, $pinned_components);
        }
      }
    }

    return $build;
  }

  /**
   * Build a single component preview.
   *
   * @param string $plugin_id
   *   The component plugin ID.
   * @param array $definition
   *   The component definition.
   * @param array $pinned_components
   *   Array of pinned component plugin IDs.
   *
   * @return array
   *   Render array for the component preview.
   */
  protected function buildComponentPreview($plugin_id, array $definition, array $pinned_components = []) {
    $component_name = $definition['name'] ?? $plugin_id;
    $is_pinned = in_array($plugin_id, $pinned_components);

    $build = [
      '#type' => 'details',
      '#title' => $component_name,
      '#open' => TRUE,
      '#attributes' => [
        'class' => ['cl-preview-component-wrapper'],
        'data-component-id' => $plugin_id,
      ],
    ];

    // Component metadata.
    if (!empty($definition['description'])) {
      $build['description'] = [
        '#type' => 'item',
        '#markup' => $definition['description'],
      ];
    }

    // Preview section.
    $build['preview'] = [
      '#type' => 'fieldset',
      '#attributes' => ['class' => ['cl-preview-iframe-container']],
    ];

    $iframe_url = Url::fromRoute('cl_preview.iframe', ['plugin_id' => $plugin_id])->toString();

    // Loading state wrapper.
    $build['preview']['wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['cl-preview-iframe-wrapper']],
    ];

    $build['preview']['wrapper']['loading'] = [
      '#type' => 'markup',
      '#markup' => '<div class="cl-preview-loading">Loading preview...</div>',
    ];

    // Lazy-loaded iframe with error handling.
    $build['preview']['wrapper']['iframe'] = [
      '#type' => 'inline_template',
      '#template' => '<iframe data-lazy-src="{{ url }}" data-component-id="{{ plugin_id }}" class="cl-preview-iframe" frameborder="0" onload="this.parentElement.querySelector(\'.cl-preview-loading\').style.display = \'none\';" onerror="this.parentElement.querySelector(\'.cl-preview-loading\').innerHTML = \'<div class=\\\'messages messages--error\\\'>Error loading component preview</div>\';"></iframe>',
      '#context' => [
        'url' => $iframe_url,
        'plugin_id' => $plugin_id,
      ],
    ];

    // Get the snippet being used for this component.
    $snippet = $this->getSnippetForComponent($plugin_id, $definition);
    $snippet_source = $this->componentHelper->hasExamplesTag($definition) ? 'From @examples in Twig template' : 'Auto-generated from component props';

    // Twig Code Snippet section (collapsed by default).
    $build['code_snippet'] = [
      '#type' => 'details',
      '#title' => $this->t('Twig Code Snippet'),
      '#open' => FALSE,
      '#attributes' => ['class' => ['cl-preview-code-snippet']],
    ];

    $build['code_snippet']['source'] = [
      '#type' => 'markup',
      '#markup' => '<p><em>' . $snippet_source . '</em></p>',
    ];

    $build['code_snippet']['code'] = [
      '#type' => 'markup',
      '#markup' => '<pre><code>' . htmlspecialchars($snippet) . '</code></pre>',
    ];

    return $build;
  }

  /**
   * Get the snippet being used for a component.
   *
   * @param string $plugin_id
   *   The component plugin ID.
   * @param array $definition
   *   The component definition.
   *
   * @return string
   *   The Twig snippet.
   */
  protected function getSnippetForComponent($plugin_id, array $definition) {
    // Try to get the first example from @examples tag.
    $example_snippet = $this->componentHelper->getFirstExampleSnippet($definition);

    if ($example_snippet) {
      return $example_snippet;
    }

    // Fallback: Generate from props.
    return $this->componentHelper->generateSnippetFromProps($plugin_id, $definition);
  }

}
