<?php

namespace Drupal\cl_preview\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\cl_preview\Service\ComponentDiscoveryService;
use Drupal\cl_preview\Service\ComponentHelperService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the component library page.
 */
class ComponentLibraryController extends ControllerBase {

  /**
   * The component discovery service.
   *
   * @var \Drupal\cl_preview\Service\ComponentDiscoveryService
   */
  protected $componentDiscovery;

  /**
   * The component helper service.
   *
   * @var \Drupal\cl_preview\Service\ComponentHelperService
   */
  protected $componentHelper;

  /**
   * Constructs a ComponentLibraryController object.
   */
  public function __construct(
    ComponentDiscoveryService $component_discovery,
    ComponentHelperService $component_helper
  ) {
    $this->componentDiscovery = $component_discovery;
    $this->componentHelper = $component_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cl_preview.component_discovery'),
      $container->get('cl_preview.component_helper')
    );
  }

  /**
   * Builds the component library page.
   */
  public function build() {
    $config = $this->config('cl_preview.settings');
    $scan_themes = $config->get('scan_themes') ?: [];
    $scan_modules = $config->get('scan_modules') ?: [];

    // Check if no themes or modules are configured.
    if (empty($scan_themes) && empty($scan_modules)) {
      $settings_url = Url::fromRoute('cl_preview.settings');

      $build['empty_message'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No themes or modules are configured for scanning. Please <a href="@settings_url">configure the module settings</a> to select which themes and modules should be scanned for components.', [
            '@settings_url' => $settings_url->toString(),
          ]),
        ],
      ];

      return $build;
    }

    $grouped_components = $this->componentDiscovery->getGroupedComponents();

    // Check if no components were found even though themes/modules are configured.
    if (empty($grouped_components)) {
      $build['no_components'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('No components found in the configured themes and modules. Make sure the selected themes/modules contain a <code>components/</code> directory with <code>*.component.yml</code> files.'),
        ],
      ];

      return $build;
    }

    // Get pinned components.
    $pinned_components = $config->get('pinned_components') ?: [];

    $build['#attached']['library'][] = 'cl_preview/library';

    $build['intro'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('Select a component to view its preview. Use the star icon to pin components to the All-Components page.'),
    ];

    // Group components by provider.
    $by_provider = $this->componentHelper->groupComponentsByProvider($grouped_components);

    // Build navigation groups organized by provider.
    foreach ($by_provider as $provider => $groups) {
      $provider_key = 'provider_' . $this->componentHelper->generateKey($provider);
      $provider_label = $this->componentHelper->getProviderLabel($provider);

      foreach ($groups as $group => $components) {
        $group_key = $provider_key . '_' . $this->componentHelper->generateKey($group);

        $items = [];
        foreach ($components as $plugin_id => $definition) {
          $is_pinned = in_array($plugin_id, $pinned_components);

          // Build item with pin icon and link.
          $items[] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['cl-preview-library-item']],
            'pin' => $this->formBuilder()->getForm(
              'Drupal\cl_preview\Form\PinComponentForm',
              $plugin_id,
              $is_pinned
            ),
            'link' => [
              '#type' => 'link',
              '#title' => ($is_pinned ? '⭐ ' : '') . ($definition['name'] ?? $plugin_id),
              '#url' => Url::fromRoute('cl_preview.component', ['plugin_id' => $plugin_id]),
            ],
          ];
        }

        $build[$group_key] = [
          '#type' => 'details',
          '#title' => $provider_label . ' (' . $group . ')',
          '#open' => TRUE,
          'list' => [
            '#theme' => 'item_list',
            '#items' => $items,
            '#list_type' => 'ul',
          ],
        ];
      }
    }

    return $build;
  }

}
